<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Server;
use App\Models\ServerIp;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class SyncVultrNetworkInfo extends Command
{
    protected $signature = 'vultr:sync-network';
    protected $description = 'همگام‌سازی اطلاعات شبکه (Netmask, Gateway, rDNS) سرورهای Vultr';

    private $apiKey;

    public function __construct()
    {
        parent::__construct();
        $this->apiKey = config('services.vultr.api_key');
    }

    public function handle()
    {
        $this->info('🔄 شروع همگام‌سازی اطلاعات شبکه Vultr...');

        // دریافت سرورهای Vultr فعال
        $servers = Server::where('provider', 'vultr')
            ->whereIn('status', ['active', 'running', 'creating'])
            ->get();

        if ($servers->isEmpty()) {
            $this->warn('⚠️  سروری یافت نشد!');
            return 0;
        }

        $updated = 0;
        $errors = 0;

        foreach ($servers as $server) {
            try {
                $this->info("📡 دریافت اطلاعات سرور: {$server->name} (ID: {$server->provider_id})");

                // دریافت اطلاعات از API
                $response = Http::withHeaders([
                    'Authorization' => 'Bearer ' . $this->apiKey,
                ])->get("https://api.vultr.com/v2/instances/{$server->provider_id}");

                if (!$response->successful()) {
                    $this->error("❌ خطا در دریافت اطلاعات سرور {$server->name}");
                    $errors++;
                    continue;
                }

                $instance = $response->json()['instance'] ?? [];

                // به‌روزرسانی IP اصلی سرور
                if ($server->ip_address !== ($instance['main_ip'] ?? null)) {
                    $server->ip_address = $instance['main_ip'] ?? null;
                    $server->save();
                    $this->info("  ✅ IP اصلی سرور به‌روز شد: {$server->ip_address}");
                }

                // به‌روزرسانی Primary IPv4
                $primaryIPv4 = ServerIp::where('server_id', $server->id)
                    ->where('type', 'ipv4')
                    ->where('is_primary', 1)
                    ->first();

                if ($primaryIPv4) {
                    $updated_fields = [];
                    
                    if ($primaryIPv4->ip !== ($instance['main_ip'] ?? null)) {
                        $primaryIPv4->ip = $instance['main_ip'] ?? null;
                        $updated_fields[] = 'IP';
                    }
                    
                    if ($primaryIPv4->netmask !== ($instance['netmask_v4'] ?? null)) {
                        $primaryIPv4->netmask = $instance['netmask_v4'] ?? null;
                        $updated_fields[] = 'Netmask';
                    }
                    
                    if ($primaryIPv4->gateway !== ($instance['gateway_v4'] ?? null)) {
                        $primaryIPv4->gateway = $instance['gateway_v4'] ?? null;
                        $updated_fields[] = 'Gateway';
                    }

                    if (!empty($updated_fields)) {
                        $primaryIPv4->save();
                        $this->info("  ✅ IPv4 به‌روز شد: " . implode(', ', $updated_fields));
                        $updated++;
                    }
                } else if (isset($instance['main_ip']) && $instance['main_ip'] !== '0.0.0.0') {
                    // اگه IPv4 وجود نداره، ایجاد کن
                    ServerIp::create([
                        'server_id' => $server->id,
                        'user_id' => $server->user_id,
                        'ip' => $instance['main_ip'],
                        'type' => 'ipv4',
                        'is_primary' => 1,
                        'is_floating' => 0,
                        'assigned_at' => now(),
                        'status' => 'active',
                        'provider' => 'vultr',
                        'location' => $server->region,
                        'hourly_cost' => 0,
                        'netmask' => $instance['netmask_v4'] ?? null,
                        'gateway' => $instance['gateway_v4'] ?? null,
                    ]);
                    $this->info("  ✅ IPv4 جدید ایجاد شد: {$instance['main_ip']}");
                    $updated++;
                }

                // به‌روزرسانی Primary IPv6
                $primaryIPv6 = ServerIp::where('server_id', $server->id)
                    ->where('type', 'ipv6')
                    ->where('is_primary', 1)
                    ->first();

                if ($primaryIPv6 && isset($instance['v6_main_ip']) && $instance['v6_main_ip'] !== '') {
                    $updated_v6 = false;
                    
                    if ($primaryIPv6->ip !== $instance['v6_main_ip']) {
                        $primaryIPv6->ip = $instance['v6_main_ip'];
                        $updated_v6 = true;
                    }
                    
                    if ($primaryIPv6->netmask !== ($instance['v6_network'] ?? null)) {
                        $primaryIPv6->netmask = $instance['v6_network'] ?? null;
                        $updated_v6 = true;
                    }

                    if ($updated_v6) {
                        $primaryIPv6->save();
                        $this->info("  ✅ IPv6 به‌روز شد");
                    }
                } else if (!$primaryIPv6 && isset($instance['v6_main_ip']) && $instance['v6_main_ip'] !== '') {
                    // ایجاد IPv6
                    ServerIp::create([
                        'server_id' => $server->id,
                        'user_id' => $server->user_id,
                        'ip' => $instance['v6_main_ip'],
                        'type' => 'ipv6',
                        'is_primary' => 1,
                        'is_floating' => 0,
                        'assigned_at' => now(),
                        'status' => 'active',
                        'provider' => 'vultr',
                        'location' => $server->region,
                        'hourly_cost' => 0,
                        'netmask' => $instance['v6_network'] ?? null,
                    ]);
                    $this->info("  ✅ IPv6 جدید ایجاد شد: {$instance['v6_main_ip']}");
                    $updated++;
                }

            } catch (\Exception $e) {
                $this->error("❌ خطا در پردازش سرور {$server->name}: {$e->getMessage()}");
                Log::error('Vultr Sync Network Error', [
                    'server_id' => $server->id,
                    'error' => $e->getMessage(),
                ]);
                $errors++;
            }
        }

        $this->info("\n✅ همگام‌سازی تمام شد!");
        $this->info("📊 تعداد به‌روزرسانی: {$updated}");
        $this->info("❌ تعداد خطا: {$errors}");

        return 0;
    }
}