<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\Server;
use App\Models\User;
use App\Mail\ServerCreatedMail;
use Illuminate\Support\Facades\DB;
use App\Services\LimitService;
use App\Services\WalletService;

class ArvanController extends Controller
{
    private $apiToken;
    private $markup = 1.3;         // سود درصدی روی قیمت ساعت
    private $min_balance = 70000;  // حداقل موجودی کیف پول برای ساخت سرور

    public function __construct(LimitService $limitService, WalletService $walletService)
    {
        $this->limitService = $limitService;
        $this->walletService = $walletService;
        $this->apiToken = env('ARVAN_API_TOKEN', '4aa5e39e-b99a-5246-81d1-7d055124cdfc');
    }

    // لیست مناطق (Regions)
    public function regions()
    {
        $response = Http::withHeaders([
            'Authorization' => 'Apikey ' . $this->apiToken,
            'Accept'        => 'application/json',
        ])->get('https://napi.arvancloud.ir/ecc/v1/regions');

        if ($response->successful()) {
            $data = $response->json();
            $locations = collect($data['data'])->filter(function ($region) {
                return isset($region['create'], $region['visible']) && $region['create'] && $region['visible'];
            })->map(function ($region) {
                return [
                    'code'    => $region['code'],
                    'name'    => "{$region['city']} ({$region['dc']})",
                    'city'    => $region['city'],
                    'dc'      => $region['dc'],
                    'country' => $region['country'],
                ];
            })->values();
            return response()->json(['status' => 'success', 'regions' => $locations]);
        }
        return response()->json(['status' => 'error', 'message' => 'خطا در دریافت regions از سرور'], $response->status());
    }

    // دریافت پلن‌های هر region (یک متد کافی است)
    public function plansByRegion($region)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Apikey ' . $this->apiToken,
                'Accept'        => 'application/json',
            ])->get("https://napi.arvancloud.ir/ecc/v1/regions/{$region}/sizes");

            if ($response->successful()) {
                $data = $response->json();
                return response()->json(['status' => 'success', 'plans' => $data['data'] ?? []]);
            }
            return response()->json([
                'status' => 'error',
                'message' => 'خطا در دریافت پلن از آروان',
                'arvan_status' => $response->status(),
                'arvan_body' => $response->body()
            ], $response->status());
        } catch (\Exception $e) {
            Log::error('Arvan plansByRegion Exception: '.$e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'خطا در سرور: ' . $e->getMessage()
            ], 500);
        }
    }

    // دریافت ایمیج‌های هر region (فقط توزیع‌های اصلی)
    public function imagesByRegion($region)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Apikey ' . $this->apiToken,
                'Accept'        => 'application/json',
            ])->get("https://napi.arvancloud.ir/ecc/v1/regions/{$region}/images?type=distributions");

            if ($response->successful()) {
                $data = $response->json();
                return response()->json(['status' => 'success', 'images' => $data['data'] ?? []]);
            }
            return response()->json([
                'status' => 'error',
                'message' => 'خطا در دریافت ایمیج‌های آروان',
                'arvan_status' => $response->status(),
                'arvan_body' => $response->body()
            ], $response->status());
        } catch (\Exception $e) {
            Log::error('Arvan imagesByRegion Exception: '.$e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'خطا در سرور: ' . $e->getMessage()
            ], 500);
        }
    }


public function vnc(Request $request, Server $server)
{
    $arvanToken = config('services.arvan.token');
    $region     = $server->region;
    $serverId   = $server->provider_id;

    $response = \Illuminate\Support\Facades\Http::withHeaders([
        'Authorization' => 'Apikey ' . $arvanToken,
        'Content-Type'  => 'application/json',
        'Accept'        => 'application/json',
   ])->get("https://napi.arvancloud.ir/ecc/v1/regions/{$region}/servers/{$serverId}/vnc");

    if ($response->successful()) {
        $data = $response->json();
        return response()->json([
            'success' => true,
            'data' => $data
        ]);
    } else {
        return response()->json([
            'success' => false,
            'message' => $response->body()
        ], $response->status());
    }
}


public function images(Request $request)
{
    $arvanToken = config('services.arvan.token');
    $region = $request->get('region', 'ir-thr-fr1');
    $type = $request->get('type', 'distributions'); // پیش‌فرض "distributions"

    $images = [];
    $response = \Illuminate\Support\Facades\Http::withHeaders([
        'Authorization' => 'Apikey ' . $arvanToken,
        'Accept'        => 'application/json',
    ])->get("https://napi.arvancloud.ir/ecc/v1/regions/{$region}/images", [
        'type' => $type, // حتماً پارامتر type
    ]);
    if ($response->successful()) {
        $images = $response->json('data') ?? [];
    }
    return response()->json($images);
}

    // دریافت شبکه‌های هر region
    public function networksByRegion($region)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Apikey ' . $this->apiToken,
                'Accept'        => 'application/json',
            ])->get("https://napi.arvancloud.ir/ecc/v1/regions/{$region}/networks");

            if ($response->successful()) {
                $data = $response->json();
                return response()->json(['status' => 'success', 'networks' => $data['data'] ?? []]);
            }
            return response()->json([
                'status' => 'error',
                'message' => 'خطا در دریافت شبکه‌ها از آروان',
                'arvan_status' => $response->status(),
                'arvan_body' => $response->body()
            ], $response->status());
        } catch (\Exception $e) {
            Log::error('Arvan networksByRegion Exception: '.$e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'خطا در سرور: ' . $e->getMessage()
            ], 500);
        }
    }



public function rebootVM(Request $request, $serverId)
{
    // فرض بر اینکه regional slug هم نیاز هست (اگر در مدل ذخیره کردی)
    // اگر توی request یا دیتابیس region رو داری، استخراج کن:
    $server = Server::where('provider', 'arvan')->where('provider_id', $serverId)->first();
    if(!$server) {
        return response()->json(['status' => 'error', 'message' => 'سرور پیدا نشد!'], 404);
    }
    $region = $server->region;

    $arvanToken = config('services.arvan.token');
    $response = \Illuminate\Support\Facades\Http::withHeaders([
        'Authorization' => 'Apikey ' . $arvanToken,
        'Accept'        => 'application/json',
    ])->post("https://napi.arvancloud.ir/ecc/v1/regions/{$region}/servers/{$serverId}/reboot");

    if ($response->successful()) {
        $server->status = 'rebooting';
        $server->save();
        return response()->json(['status' => 'success', 'message' => 'درخواست ریبوت ارسال شد.']);
    }

    // اگر نیاز داشتی خطای بیشتر را log کن یا به کاربر برگردان
    return response()->json(['status' => 'error', 'message' => 'خطا در ریبوت سرور.'], 500);
}


// ساخت سرور جدید
public function createVM(Request $request, $projectId = null)
{
    Log::info('ARVAN-createVM: درخواست دریافتی', [
        'project_id' => $projectId,
        'all' => $request->all(),
    ]);

    $data = $request->validate([
        'region' => 'required|string',
        'plan'   => 'required|string',
        'image'  => 'required|string',
    ]);
    Log::info('ARVAN-createVM: داده validate شده', $data);

    $user = User::find(auth()->id());
    if (!$user || $user->wallet_balance < $this->min_balance) {
        Log::warning('NOT_ENOUGH_BALANCE_BLOCKED_ARVAN', [
            'wallet_balance' => $user ? $user->wallet_balance : null,
            'min_balance' => $this->min_balance,
        ]);
        return response()->json([
            'status'  => 'error',
            'message' => 'برای ساخت سرور ایران باید حداقل ' . number_format($this->min_balance) . ' تومان موجودی کیف پول داشته باشید.'
        ], 422);
    }
    
    // ✅ چک محدودیت سرورها
    if (!$this->limitService->canCreateServer($user->id)) {
        $info = $this->limitService->getLimitInfo($user->id);
        return response()->json([
            'status' => 'error',
            'message' => 'شما به حد محدودیت سرورهای مجاز رسیده‌اید',
            'limit_info' => $info['servers'],
            'current_servers' => $info['servers']['used'],
            'max_servers' => $info['servers']['limit']
        ], 403);
    }

    // دریافت قیمت و مشخصات پلن
    $plan_id = $data['plan'];
    $regionCode = $data['region'];
    $planApiResponse = Http::withHeaders([
        'Authorization' => 'Apikey ' . $this->apiToken,
        'Accept'        => 'application/json',
    ])->get("https://napi.arvancloud.ir/ecc/v1/regions/{$regionCode}/sizes");

    $priceHourlyRial = null;
    $planDisk = 20;

    if ($planApiResponse->successful()) {
        $plans = $planApiResponse->json('data') ?? [];
        foreach ($plans as $plan) {
            if ($plan['id'] == $plan_id) {
                $priceHourlyRial = (int)($plan['price_per_hour'] ?? 0); // قیمت ریالی
                $planDisk = (int)($plan['disk'] ?? 20);
                break;
            }
        }
    }

    if (!$priceHourlyRial || $priceHourlyRial <= 0) {
        return response()->json([
            'status' => 'error',
            'message' => 'قیمت پلن از آروان دریافت نشد!'
        ], 422);
    }

    // 1) ریال به تومان
    $priceHourlyToman = $priceHourlyRial / 10;

    // 2) افزودن 20 درصد (تومان)
    $priceHourlyToman = (int) ceil($priceHourlyToman * 1.20);

    // دریافت اولین شبکه فعال منطقه
    $networkResp = Http::withHeaders([
        'Authorization' => 'Apikey ' . $this->apiToken,
        'Accept'        => 'application/json',
    ])->get("https://napi.arvancloud.ir/ecc/v1/regions/{$regionCode}/networks");
    $network_id = $networkResp->json('data.0.id') ?? null;
    if (!$network_id) {
        return response()->json(['status'=>'error','message'=>'نتورک برای این موقعیت پیدا نشد!'], 422);
    }

    $serverName = 'IrOnCloud-' . uniqid();
    $payload = [
        'name'          => $serverName,
        'flavor_id'     => $plan_id,
        'image_id'      => $data['image'],
        'network_id'    => $network_id,
        'disk_size'     => $planDisk,
        'ssh_key'       => true,
        'enable_ipv4'   => true,
        'enable_ipv6'   => false,
        'create_type'   => 'cinder',
        'enable_backup' => false,
        'ha_enabled'    => false
    ];

    Log::info('ARVAN-createVM: payload', $payload);

    $response = Http::withHeaders([
        'Authorization' => 'Apikey ' . $this->apiToken,
        'Accept'        => 'application/json',
    ])->post("https://napi.arvancloud.ir/ecc/v1/regions/{$regionCode}/servers", $payload);

    Log::info('ARVAN-createVM: پاسخ از آروان', [
        'status_code' => $response->status(),
        'body' => $response->body()
    ]);

    if ($response->successful()) {
        $instance = $response->json('data');
        Log::info('ARVAN-createVM: instance دریافتی', $instance);

        DB::beginTransaction();
        try {
            $server = Server::create([
                'user_id'      => $user->id,
                'project_id'   => $projectId,
                'name'         => $serverName,
                'region'       => $regionCode,
                'plan'         => $plan_id,
                'password'     => $instance['password'] ?? null,
                'image'        => $data['image'],
                'status'       => 'creating',
                'provider'     => 'arvan',
                'provider_id'  => $instance['id'] ?? null,
                'ip_address'   => null, // بعداً با sync پر می‌شه
                'price_hourly_toman' => round($priceHourlyToman),
            ]);

            // 💰 کسر هزینه ساعت اول با WalletService
            if ($server && $priceHourlyToman > 0) {
                $walletService = app(\App\Services\WalletService::class);
                
                $transaction = $walletService->deductServerHourlyCost(
                    $user,
                    $server->id,
                    $server->name,
                    round($priceHourlyToman),
                    'arvan'
                );

                if ($transaction) {
                    Log::info('💰 First hour cost deducted (Arvan)', [
                        'server_id' => $server->id,
                        'amount' => round($priceHourlyToman),
                        'transaction_id' => $transaction->id,
                    ]);
                } else {
                    // موجودی کافی نیست - باید سرور رو حذف کنیم یا suspend کنیم
                    Log::error('⚠️ Insufficient balance for first hour - Rolling back', [
                        'server_id' => $server->id,
                        'required_amount' => round($priceHourlyToman),
                        'user_balance' => $user->wallet_balance,
                    ]);
                    
                    DB::rollBack();
                    
                    // اختیاری: حذف سرور از آروان
                    // $this->deleteServerFromArvan($instance['id'], $regionCode);
                    
                    return response()->json([
                        'status' => 'error',
                        'message' => 'موجودی کیف پول برای پرداخت هزینه ساعت اول کافی نیست.'
                    ], 422);
                }
            }

            // ✅ ذخیره Primary IPv4 (اگر الان موجود باشه)
            // معمولاً آروان IP رو بلافاصله نمیده و باید با sync گرفت
            // ولی اگر در response هست، ذخیره می‌کنیم
            if (isset($instance['networks']) && is_array($instance['networks'])) {
                foreach ($instance['networks'] as $network) {
                    if (isset($network['type']) && $network['type'] === 'public') {
                        $ipv4 = $network['ip'] ?? null;
                        if ($ipv4) {
                            \App\Models\ServerIp::create([
                                'server_id' => $server->id,
                                'user_id' => $server->user_id,
                                'ip' => $ipv4,
                                'type' => 'ipv4',
                                'is_primary' => 1,
                                'is_floating' => 0,
                                'assigned_at' => now(),
                                'status' => 'active',
                                'provider' => 'arvan',
                                'location' => $regionCode,
                                'hourly_cost' => 0, // Primary IP معمولاً رایگان
                            ]);
                            
                            // آپدیت IP در سرور
                            $server->update(['ip_address' => $ipv4]);
                            
                            Log::info('✅ Arvan Primary IPv4 saved', [
                                'server_id' => $server->id,
                                'ip' => $ipv4,
                            ]);
                        }
                    }
                }
            }

            DB::commit();

            // ارسال ایمیل (اختیاری)
            try {
                \Mail::to($user->email)->later(
                    now()->addSeconds(30),
                    new \App\Mail\ServerCreatedMail(
                        $user,
                        $server,
                        config('app.url') . '/cloud/'
                    )
                );
                
                Log::info('EMAIL_QUEUED (Arvan)', [
                    'user_email' => $user->email,
                    'server_id' => $server->id,
                ]);
            } catch (\Exception $e) {
                Log::error('EMAIL_SEND_FAILED (Arvan)', [
                    'error' => $e->getMessage(),
                    'server_id' => $server->id,
                ]);
            }

            return response()->json([
                'status'    => 'success',
                'message'   => 'سرور شما با موفقیت ساخته شد.',
                'server_id' => $server->id,
            ]);
            
        } catch (\Throwable $e) {
            DB::rollBack();
            Log::error('ARVAN-createVM: مشکل در ثبت دیتابیس - ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString(),
            ]);
            return response()->json([
                'status' => 'error',
                'message' => 'سرویس ساخته شد ولی در ثبت اطلاعات خطا رخ داد!'
            ], 500);
        }
    }

    // اگر response موفق نبود
    $error = $response->json();
    $message = $error['message'] ?? 'خطا در ساخت سرور!';
    Log::error('ARVAN-createVM: خطا', [
        'error' => $error,
        'status' => $response->status()
    ]);
    return response()->json(['status' => 'error', 'message' => $message, 'response' => $error], $response->status());
}
}