<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class LoginController extends Controller
{
    use AuthenticatesUsers;

    protected $redirectTo = '/cloud';

    public function __construct()
    {
        $this->middleware('guest')->only('showLoginForm', 'login');
        $this->middleware('auth')->only('logout');
    }

    /**
     * Show the login form
     */
    public function showLoginForm()
    {
        Log::info('showLoginForm called - Auth check: ' . (auth()->check() ? 'yes' : 'no'));
        
        if (auth()->check()) {
            Log::info('User already logged in, redirecting to dashboard');
            return redirect()->route('cloud.default_dashboard');
        }
        
        return view('auth.login');
    }
    
    /**
     * Handle login request
     */
    public function login(Request $request)
    {
        Log::info('Login attempt', ['email' => $request->email]);
        
        $credentials = $this->validate($request, [
            'email' => 'required|email',
            'password' => 'required',
        ]);

        if (auth()->attempt($credentials, $request->filled('remember'))) {
            Log::info('Login successful', ['user_id' => auth()->id()]);
            
            $request->session()->regenerate();
            
            return redirect()->intended(route('cloud.default_dashboard'))
                ->with('success', 'خوش آمدید!');
        }

        Log::warning('Login failed', ['email' => $request->email]);
        
        return back()
            ->withInput($request->only('email'))
            ->withErrors([
                'email' => 'ایمیل یا رمز عبور نادرست است.',
            ]);
    }
    
    /**
     * Redirect after successful login
     */
    protected function sendLoginResponse(Request $request)
    {
        $request->session()->regenerate();
        $this->clearLoginAttempts($request);

        return redirect()->intended($this->redirectPath());
    }
    
    /**
     * Get redirect path
     */
    public function redirectPath()
    {
        return route('cloud.default_dashboard');
    }
}