<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use App\Models\Server;
use App\Services\LimitService;
use App\Services\WalletService;
use App\Models\User;
use App\Mail\ServerCreatedMail;
use Illuminate\Support\Facades\Mail;

class HetznerController extends Controller
{
    private $apiToken;
    private $apiBase = 'https://api.hetzner.cloud/v1';
    private $euroRate = 165000; // نرخ یورو به تومان (قابل تنظیم)
    private $limitService;
    private $walletService;

    public function __construct(LimitService $limitService, WalletService $walletService)
    {
        $this->limitService = $limitService;
        $this->walletService = $walletService; 
        $this->apiToken = config('services.hetzner.api_token');
    }

    // دریافت مناطق (locations)
    public function locations()
    {
        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->apiToken,
            'Accept'        => 'application/json',
        ])->get($this->apiBase . '/locations');

        if ($response->successful()) {
            $data = $response->json();
            return response()->json([
                'status' => 'success',
                'locations' => $data['locations'] ?? [],
            ]);
        }
        return response()->json(['status' => 'error', 'message' => 'خطا در دریافت موقعیت‌های Hetzner'], $response->status());
    }

    // دریافت پلن‌ها (server types)
    public function plans()
    {
        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->apiToken,
            'Accept'        => 'application/json',
        ])->get($this->apiBase . '/server_types');

        if ($response->successful()) {
            $data = $response->json();
            return response()->json([
                'status' => 'success',
                'plans' => $data['server_types'] ?? [],
            ]);
        }
        return response()->json(['status' => 'error', 'message' => 'خطا در دریافت پلن‌های Hetzner'], $response->status());
    }

    // دریافت ایمیج‌ها (images)
    public function images()
    {
        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->apiToken,
            'Accept'        => 'application/json',
        ])->get($this->apiBase . '/images?type=system');

        if ($response->successful()) {
            $data = $response->json();
            return response()->json([
                'status' => 'success',
                'images' => $data['images'] ?? [],
            ]);
        }
        return response()->json(['status' => 'error', 'message' => 'خطا در دریافت ایمیج‌های Hetzner'], $response->status());
    }
    
// ساخت سرور جدید با cloud-init و رمز تصادفی
public function createVM(Request $request, $projectId)
{
    $data = $request->validate([
        'name'         => 'nullable|string|max:255',
        'location'     => 'required|string',
        'server_type'  => 'required|string',
        'image'        => 'required|string',
    ]);

    $user = User::find(auth()->id());
    $min_balance = 50000; // حداقل اعتبار لازم

    if (!$user || $user->wallet_balance < $min_balance) {
        return response()->json([
            'status'  => 'error',
            'message' => 'برای ساخت سرور هتزنر باید حداقل ' . number_format($min_balance) . ' تومان موجودی کیف پول داشته باشید.'
        ], 422);
    }

    if (!$this->limitService->canCreateServer($user->id)) {
        $info = $this->limitService->getLimitInfo($user->id);
        return response()->json([
            'status' => 'error',
            'message' => 'شما به حد محدودیت سرورهای مجاز رسیده‌اید',
            'limit_info' => $info['servers'],
            'current_servers' => $info['servers']['used'],
            'max_servers' => $info['servers']['limit']
        ], 403);
    }


    // دریافت قیمت پلن با توجه به دیتاسنتر انتخابی
    $planApi = Http::withHeaders([
        'Authorization' => 'Bearer ' . $this->apiToken,
        'Accept'        => 'application/json',
    ])->get($this->apiBase . '/server_types');

    $hourlyEuro = null;
    if ($planApi->successful()) {
        $plans = $planApi->json('server_types') ?? [];
        foreach ($plans as $plan) {
            if ($plan['name'] == $data['server_type']) {
                // پیدا کردن قیمت مخصوص location انتخاب‌شده
                if (isset($plan['prices']) && is_array($plan['prices'])) {
                    foreach ($plan['prices'] as $price) {
                        if (isset($price['location']) && $price['location'] == $data['location']) {
                            $hourlyEuro = $price['price_hourly']['net'] ?? null;
                            break 2;
                        }
                    }
                }
            }
        }
    }
    if (!$hourlyEuro) {
        return response()->json([
            'status' => 'error',
            'message' => 'خطا در دریافت قیمت ساعتی پلن هتزنر!'
        ], 422);
    }

    $priceHourlyToman = $hourlyEuro * $this->euroRate;
    $totalHourlyPrice = ceil($priceHourlyToman * 1.25); // +25% سود
    
    // ✅ تقسیم قیمت: سرور + IPv4 (IPv6 رایگان)
    $ipv4Cost = 250; // IPv4 Primary هزینه داره
    $ipv6Cost = 0;   // IPv6 رایگانه
    $serverOnlyCost = $totalHourlyPrice - $ipv4Cost; // قیمت سرور بدون IPv4

    // --- رمز تصادفی و cloud-init (user_data) ---
    $password = \Illuminate\Support\Str::random(14);
    $userData = <<<CLOUDINIT
#cloud-config
chpasswd:
  list: |
    root:$password
  expire: False
ssh_pwauth: True
CLOUDINIT;

    $payload = [
        'name'         => $data['name'] ?? ('hetzner-' . uniqid()),
        'server_type'  => $data['server_type'],
        'location'     => $data['location'],
        'image'        => $data['image'],
        'start_after_create' => true,
        'user_data'    => $userData,
    ];

    $response = Http::withHeaders([
        'Authorization' => 'Bearer ' . $this->apiToken,
        'Content-Type'  => 'application/json',
        'Accept'        => 'application/json',
    ])->post($this->apiBase . '/servers', $payload);

    if ($response->successful()) {
        $server = $response->json()['server'] ?? [];
        
        // ✅ Log کامل response برای دیباگ
        Log::info('Hetzner Server Response', ['server_data' => $server]);
        
        // ✅ ذخیره سرور با قیمت کمتر (بدون IPv4)
        $serverData = [
            'user_id'      => auth()->id(),
            'project_id'   => $projectId,
            'name'         => $payload['name'],
            'region'       => $data['location'],
            'status'       => 'creating',
            'provider'     => 'hetzner',
            'provider_id'  => $server['id'] ?? null,
            'plan'         => $data['server_type'],
            'image'        => $data['image'],
            'ip_address'   => $server['public_net']['ipv4']['ip'] ?? null,
            'password'     => $password,
            'price_hourly_toman' => $serverOnlyCost,
        ];
        $serverModel = Server::create($serverData);

        // متغیرهای ذخیره IP برای کسر هزینه
        $primaryIpv4 = null;
        $primaryIpv6 = null;

        // ✅ ذخیره Primary IPv4 با هزینه 250 تومان
        if (isset($server['public_net']['ipv4']['ip'])) {
            try {
                $primaryIpv4 = \App\Models\ServerIp::create([
                    'server_id' => $serverModel->id,
                    'user_id' => $serverModel->user_id,
                    'ip' => $server['public_net']['ipv4']['ip'],
                    'type' => 'ipv4',
                    'is_primary' => 1,
                    'is_floating' => 0,
                    'assigned_at' => now(),
                    'status' => 'active',
                    'provider' => 'hetzner',
                    'provider_ip_id' => $server['public_net']['ipv4']['id'] ?? null,
                    'location' => $server['datacenter']['location']['name'] ?? $data['location'],
                    'datacenter' => $server['datacenter']['name'] ?? null,
                    'hourly_cost' => $ipv4Cost,
                    'rdns' => null,
                ]);
                
                Log::info('✅ Hetzner Primary IPv4 saved', [
                    'server_id' => $serverModel->id,
                    'ip' => $server['public_net']['ipv4']['ip'],
                    'hourly_cost' => $ipv4Cost
                ]);
            } catch (\Exception $e) {
                Log::error('❌ Failed to save IPv4', [
                    'error' => $e->getMessage(),
                    'server_id' => $serverModel->id,
                ]);
            }
        }

        // ✅ ذخیره Primary IPv6 رایگان (اگه داره)
        if (isset($server['public_net']['ipv6']['ip'])) {
            try {
                $primaryIpv6 = \App\Models\ServerIp::create([
                    'server_id' => $serverModel->id,
                    'user_id' => $serverModel->user_id,
                    'ip' => $server['public_net']['ipv6']['ip'],
                    'type' => 'ipv6',
                    'is_primary' => 1,
                    'is_floating' => 0,
                    'assigned_at' => now(),
                    'status' => 'active',
                    'provider' => 'hetzner',
                    'provider_ip_id' => $server['public_net']['ipv6']['id'] ?? null,
                    'location' => $server['datacenter']['location']['name'] ?? $data['location'],
                    'datacenter' => $server['datacenter']['name'] ?? null,
                    'hourly_cost' => $ipv6Cost,
                    'rdns' => null,
                ]);
                
                Log::info('✅ Hetzner Primary IPv6 saved (FREE)', [
                    'server_id' => $serverModel->id,
                    'ip' => $server['public_net']['ipv6']['ip'],
                    'hourly_cost' => 0
                ]);
            } catch (\Exception $e) {
                Log::error('❌ Failed to save IPv6', [
                    'error' => $e->getMessage(),
                    'server_id' => $serverModel->id,
                ]);
            }
        }

        // 💰 کسر هزینه ساعت اول سرور و IP
        try {
            $walletService = app(\App\Services\WalletService::class);
            
            // کسر هزینه ساعت اول سرور
            $serverTransaction = $walletService->deductServerHourlyCost(
                $user,
                $serverModel->id,
                $serverModel->name,
                $serverOnlyCost,
                'hetzner'
            );

            if ($serverTransaction) {
                Log::info('💰 First hour server cost deducted', [
                    'server_id' => $serverModel->id,
                    'amount' => $serverOnlyCost,
                    'transaction_id' => $serverTransaction->id,
                ]);
            } else {
                Log::warning('⚠️ Failed to deduct first hour server cost - insufficient balance', [
                    'server_id' => $serverModel->id,
                    'required_amount' => $serverOnlyCost,
                    'user_balance' => $user->wallet_balance,
                ]);
            }

            // کسر هزینه ساعت اول IPv4 (اگر وجود داشته باشد)
            if ($primaryIpv4 && $ipv4Cost > 0) {
                $ipTransaction = $walletService->deductIpHourlyCost(
                    $user,
                    $primaryIpv4->id,
                    $primaryIpv4->ip,
                    $serverModel->id,
                    $ipv4Cost,
                    'ipv4'
                );

                if ($ipTransaction) {
                    Log::info('💰 First hour IPv4 cost deducted', [
                        'ip_id' => $primaryIpv4->id,
                        'ip' => $primaryIpv4->ip,
                        'amount' => $ipv4Cost,
                        'transaction_id' => $ipTransaction->id,
                    ]);
                } else {
                    Log::warning('⚠️ Failed to deduct first hour IPv4 cost - insufficient balance', [
                        'ip_id' => $primaryIpv4->id,
                        'required_amount' => $ipv4Cost,
                        'user_balance' => $user->wallet_balance,
                    ]);
                }
            }

            // IPv6 رایگان است، نیازی به کسر هزینه نیست
            
        } catch (\Exception $e) {
            Log::error('❌ Error deducting first hour costs', [
                'server_id' => $serverModel->id,
                'error' => $e->getMessage(),
            ]);
        }

        // ارسال ایمیل اطلاع‌رسانی ساخت سرور
        try {
            Mail::to($user->email)->send(new ServerCreatedMail($user, $serverModel, url('/cloud/servers/'.$serverModel->id.'/overview')));
            Log::info('EMAIL_SENT: ServerCreatedMail', [
                'user_email' => $user->email,
                'server_id' => $serverModel->id
            ]);
        } catch (\Exception $e) {
            Log::error('EMAIL_ERROR: ServerCreatedMail', [
                'user_email' => $user->email ?? null,
                'server_id' => $serverModel->id,
                'error_message' => $e->getMessage()
            ]);
        }

        return response()->json([
            'status'    => 'success',
            'message'   => 'سرور با موفقیت ساخته شد.',
            'server_id' => $serverModel->id,
            'password'  => $password,
        ]);
    }

    $error = $response->json();
    Log::error('Hetzner server create error', ['error' => $error, 'payload' => $payload]);
    $message = $error['error_message'] ?? $error['message'] ?? 'خطا در ساخت سرور (سیستم عامل انتخابی با نوع سرور سازگار نمیباشد)!';
    return response()->json(['status' => 'error', 'message' => $message, 'response' => $error], $response->status());
}

    // حذف سرور
    public function destroy($serverId)
    {
        $server = Server::where('provider', 'hetzner')->where('provider_id', $serverId)->first();
        if (!$server) {
            return response()->json(['status' => 'error', 'message' => 'سرور یافت نشد.'], 404);
        }

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->apiToken,
            'Accept'        => 'application/json',
        ])->delete($this->apiBase . '/servers/' . $serverId);

        if ($response->successful()) {
            $server->delete();
            return response()->json(['status' => 'success', 'message' => 'سرور با موفقیت حذف شد.']);
        }
        return response()->json(['status' => 'error', 'message' => 'خطا در حذف سرور Hetzner'], $response->status());
    }

    // ریبوت
    public function reboot($serverId)
    {
        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->apiToken,
            'Accept'        => 'application/json',
        ])->post($this->apiBase . "/servers/{$serverId}/actions/reboot");

        if ($response->successful()) {
            return response()->json(['status' => 'success', 'message' => 'ریستارت با موفقیت انجام شد.']);
        }
        return response()->json(['status' => 'error', 'message' => 'خطا در ریستارت سرور Hetzner!'], $response->status());
    }

    // پاور (روشن/خاموش)
    public function power(Request $request, $serverId)
    {
        $state = $request->input('state'); // 0=خاموش, 1=روشن
        $endpoint = $state ? 'poweron' : 'poweroff';

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->apiToken,
            'Accept'        => 'application/json',
        ])->post($this->apiBase . "/servers/{$serverId}/actions/{$endpoint}");

        if ($response->successful()) {
            return response()->json(['status' => 'success', 'message' => 'عملیات پاور انجام شد.']);
        }
        return response()->json(['status' => 'error', 'message' => 'خطا در تغییر پاور سرور!'], $response->status());
    }

    // وضعیت سرور (sync با دیتابیس)
    public function status($serverId)
    {
        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->apiToken,
            'Accept'        => 'application/json',
        ])->get($this->apiBase . "/servers/{$serverId}");

        if ($response->successful()) {
            $serverData = $response->json()['server'] ?? [];
            // دیتابیس را sync کن اگر رکورد وجود دارد
            $server = Server::where('provider', 'hetzner')->where('provider_id', $serverId)->first();
            if ($server) {
                if (!empty($serverData['status']) && $server->status !== 'suspended') {
                    $server->status = $serverData['status'];
                }
                if (!empty($serverData['public_net']['ipv4']['ip'])) {
                    $server->ip_address = $serverData['public_net']['ipv4']['ip'];
                }
                $server->save();
            }
            return response()->json([
                'status' => $serverData['status'] ?? 'unknown',
                'ip_address' => $serverData['public_net']['ipv4']['ip'] ?? '-',
                'plan' => $serverData['server_type']['name'] ?? '-',
                'image' => $serverData['image']['id'] ?? '-',
                'region' => $serverData['datacenter']['location']['name'] ?? '-'
            ]);
        }
        return response()->json(['status' => 'error', 'message' => 'خطا در دریافت وضعیت سرور Hetzner!'], $response->status());
    }
}