<?php

namespace App\Http\Controllers;

use App\Models\Server;
use App\Models\ServerIp;
use App\Services\IpManagementService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class IpController extends Controller
{
    protected $ipService;

    public function __construct(IpManagementService $ipService)
    {
        $this->ipService = $ipService;
    }

    /**
     * لیست IP های سرور
     */
    public function index(Server $server)
    {
        $this->authorize('view', $server);

        $ips = $server->ips()->orderBy('is_primary', 'desc')->get();

        return response()->json([
            'success' => true,
            'ips' => $ips->map(function($ip) {
                return [
                    'id' => $ip->id,
                    'ip' => $ip->ip,
                    'type' => $ip->type,
                    'type_label' => $ip->getTypeLabel(),
                    'is_primary' => $ip->isPrimary(),
                    'status' => $ip->status,
                    'rdns' => $ip->rdns,
                    'cost_description' => $ip->getCostDescription(),
                    'days_active' => $ip->getDaysActive(),
                    'deletable' => $ip->isDeletable(),
                    'deletion_status' => $ip->getDeletionStatus(),
                    'created_at' => $ip->created_at->format('Y-m-d H:i'),
                ];
            })
        ]);
    }

    /**
     * افزودن IP جدید (Floating IP)
     */
    public function store(Request $request, Server $server)
    {
        $this->authorize('update', $server);

        $request->validate([
            'type' => 'required|in:ipv4,ipv6'
        ]);

        try {
            // بررسی امکان افزودن
            $check = $this->ipService->canAddFloatingIp($server, $request->type);
            
            if (!$check['can_add']) {
                return response()->json([
                    'success' => false,
                    'message' => 'امکان افزودن IP وجود ندارد',
                    'errors' => $check['errors']
                ], 422);
            }

            // ایجاد Floating IP
            $ip = $this->ipService->createFloatingIp($server, $request->type);

            return response()->json([
                'success' => true,
                'message' => 'Floating IP با موفقیت اضافه شد',
                'ip' => [
                    'id' => $ip->id,
                    'ip' => $ip->ip,
                    'type' => $ip->type,
                    'type_label' => $ip->getTypeLabel(),
                    'cost_description' => $ip->getCostDescription(),
                ],
                'monthly_cost' => $check['monthly_cost']
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to add IP', [
                'server_id' => $server->id,
                'type' => $request->type,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }


/**
 * دریافت لیست IP های Detached
 */
/**
 * دریافت لیست IP های Detached
 */
public function getDetachedIps(Server $server)
{
    $this->authorize('view', $server);
    
    try {
        // دریافت IP های detached از همین کاربر
        $detachedIps = ServerIp::where('user_id', auth()->id())
            ->where('status', 'detached')
            ->where('is_primary', 1)
            ->where('type', 'ipv4')
            ->whereNull('deleted_at')
            ->orderBy('detached_at', 'desc')
            ->get()
            ->map(function($ip) {
                return [
                    'id' => $ip->id,
                    'ip' => $ip->ip,
                    'type' => $ip->type,
                    'detached_at' => $ip->detached_at ? $ip->detached_at->format('Y/m/d H:i') : '-',
                    'provider_ip_id' => $ip->provider_ip_id,
                ];
            });
        
        Log::info('Detached IPs retrieved', [
            'user_id' => auth()->id(),
            'count' => $detachedIps->count(),
        ]);
        
        return response()->json([
            'success' => true,
            'ips' => $detachedIps,
        ]);
    } catch (\Exception $e) {
        Log::error('Failed to get detached IPs', [
            'error' => $e->getMessage(),
            'user_id' => auth()->id(),
        ]);
        
        return response()->json([
            'success' => false,
            'message' => $e->getMessage(),
            'ips' => [],
        ], 500);
    }
}


    /**
     * حذف IP (فقط Floating IPs)
     */
    public function destroy(Server $server, ServerIp $ip)
    {
        $this->authorize('update', $server);

        if ($ip->server_id !== $server->id) {
            abort(403, 'دسترسی غیرمجاز');
        }

        try {
            // بررسی قابلیت حذف
            if (!$ip->isDeletable()) {
                $status = $ip->getDeletionStatus();
                return response()->json([
                    'success' => false,
                    'message' => implode(', ', $status['reasons'])
                ], 422);
            }

            // حذف IP (بدون جریمه - چون ماهانه هستند)
            $this->ipService->deleteFloatingIp($ip);

            return response()->json([
                'success' => true,
                'message' => 'IP با موفقیت حذف شد (بدون جریمه)',
                'penalty' => null // Floating IPs جریمه ندارند
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to delete IP', [
                'ip_id' => $ip->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * تغییر Primary IP (فقط Hetzner)
     */
    public function changePrimaryIp(Request $request, Server $server, ServerIp $ip)
    {
        $this->authorize('update', $server);

        if ($ip->server_id !== $server->id) {
            abort(403, 'دسترسی غیرمجاز');
        }

        // بررسی نوع IP
        if (!$ip->isPrimary()) {
            return response()->json([
                'success' => false,
                'message' => 'فقط Primary IP قابل تغییر است'
            ], 422);
        }

        try {
            // بررسی امکان تغییر
            $check = $this->ipService->canChangePrimaryIp($server);
            
            if (!$check['can_change']) {
                return response()->json([
                    'success' => false,
                    'message' => 'امکان تغییر IP وجود ندارد',
                    'errors' => $check['errors']
                ], 422);
            }

            // تغییر IP
            $newIp = $this->ipService->changePrimaryIp($server, $request->new_ip_id);

            return response()->json([
                'success' => true,
                'message' => 'Primary IP با موفقیت تغییر کرد',
                'new_ip' => [
                    'id' => $newIp->id,
                    'ip' => $newIp->ip,
                    'type' => $newIp->type,
                ],
                'change_fee' => ServerIp::IP_CHANGE_FEE,
                'remaining_changes' => auth()->user()->getRemainingIpChanges(),
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to change primary IP', [
                'server_id' => $server->id,
                'ip_id' => $ip->id,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'success' => false,
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * بررسی امکان افزودن Floating IP
     */
    public function checkCanAdd(Server $server, Request $request)
    {
        $this->authorize('view', $server);

        $request->validate([
            'type' => 'required|in:ipv4,ipv6'
        ]);

        $check = $this->ipService->canAddFloatingIp($server, $request->type);

        return response()->json([
            'success' => true,
            'can_add' => $check['can_add'],
            'errors' => $check['errors'],
            'monthly_cost' => $check['monthly_cost'],
            'monthly_cost_formatted' => number_format($check['monthly_cost']) . ' تومان/ماه'
        ]);
    }

    /**
     * بررسی امکان تغییر Primary IP
     */
    public function checkCanChange(Server $server)
    {
        $this->authorize('view', $server);

        $check = $this->ipService->canChangePrimaryIp($server);

        return response()->json([
            'success' => true,
            'can_change' => $check['can_change'],
            'errors' => $check['errors'] ?? [],
            'change_fee' => $check['change_fee'] ?? ServerIp::IP_CHANGE_FEE,
            'remaining_changes' => $check['remaining_changes'] ?? 0,
        ]);
    }

    /**
     * دریافت هزینه تخمینی حذف (Floating IP - بدون جریمه)
     */
    public function checkDeletionCost(Server $server, ServerIp $ip)
    {
        $this->authorize('view', $server);

        if ($ip->server_id !== $server->id) {
            abort(403);
        }

        // Floating IPs جریمه ندارند
        $penalty = 0;
        $deletionStatus = $ip->getDeletionStatus();

        return response()->json([
            'success' => true,
            'deletable' => $deletionStatus['deletable'],
            'has_penalty' => false, // همیشه false
            'penalty' => 0,
            'penalty_formatted' => '0 تومان',
            'days_active' => $ip->getDaysActive(),
            'reasons' => $deletionStatus['reasons'] ?? [],
            'warning' => 'Floating IP ها ماهانه هستند و جریمه حذف ندارند'
        ]);
    }

    /**
     * جداسازی IP اصلی (Deprecated - از changePrimaryIp استفاده کنید)
     */
    public function detachPrimary(Server $server, ServerIp $ip)
    {
        $this->authorize('update', $server);

        if ($ip->server_id !== $server->id) {
            abort(403, 'دسترسی غیرمجاز');
        }

        return response()->json([
            'success' => false,
            'message' => 'برای تغییر Primary IP از گزینه "تغییر IP" استفاده کنید (هزینه: 3,700 تومان)'
        ], 422);
    }
}