<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\Server;
use App\Models\User;

class OneProviderController extends Controller
{
    private $apiKey;
    private $clientKey;
    private $dollarRate = 125000; // نرخ ثابت یورو/دلار به تومان (در صورت نیاز تغییر بده)

    public function __construct()
    {
        $this->apiKey    = config('services.oneprovider.api_key');
        $this->clientKey = config('services.oneprovider.client_key');
    }

    /**
     * Helper for OneProvider API call
     */
    private function callApi($method, $endpoint, $params = [])
    {
        $url = 'https://api.oneprovider.com' . $endpoint;

        $headers = [
            'Api-Key'      => $this->apiKey,
            'Client-Key'   => $this->clientKey,
            'X-Pretty-JSON'=> '1',
            'User-Agent'   => 'OneApi/1.0',
        ];

        if (strtoupper($method) === 'GET') {
            $response = Http::withHeaders($headers)->get($url, $params);
        } else {
            $response = Http::withHeaders($headers)->asForm()->post($url, $params);
        }

        return $response;
    }

    // لیست دیتاسنترها (Locations)
    public function locations()
    {
        $response = $this->callApi('GET', '/vm/locations');
        return response()->json($response->json());
    }

    // لیست پلن‌ها (Sizes)
    public function plans()
    {
        $response = $this->callApi('GET', '/vm/sizes');
        return response()->json($response->json());
    }

    // لیست سیستم عامل‌ها (Templates)
    public function images()
    {
        $response = $this->callApi('GET', '/vm/templates');
        return response()->json($response->json());
    }

    // لیست سرورهای کاربر
    public function servers()
    {
        $response = $this->callApi('GET', '/server/list');
        return response()->json($response->json());
    }

    // مشاهده اطلاعات یک سرور خاص
    public function serverInfo($server_id)
    {
        $response = $this->callApi('GET', "/server/info/$server_id");
        return response()->json($response->json());
    }

    // ساخت ماشین مجازی جدید
    public function createVM(Request $request, $projectId = null)
    {
        $user = User::find(auth()->id());
        $min_balance = 900000;
        if (!$user || $user->wallet_balance < $min_balance) {
            return response()->json([
                'status'  => 'error',
                'message' => 'برای ساخت سرور این نوع باید حداقل ' . number_format($min_balance) . ' تومان موجودی داشته باشید.'
            ], 422);
        }

        $data = $request->validate([
            'location_id'   => 'required|string',
            'instance_size' => 'required|string',
            'template'      => 'required|string',
            'hostname'      => 'nullable|string|max:64',
        ]);

        // گرفتن قیمت پلن
        $plans = $this->callApi('GET', '/vm/sizes')->json()['response'] ?? [];
        $plan = collect($plans)->firstWhere('id', $data['instance_size']);
        if (!$plan) {
            return response()->json(['status' => 'error', 'message' => 'پلن انتخابی یافت نشد!'], 422);
        }
        $hourly_price_eur = (float)($plan['hourly_price'] ?? 0.00); // OneProvider یورو
        $price_hourly_toman = ceil($hourly_price_eur * $this->dollarRate * 1.30); // +30% سود

        // اگر اعتبار کافی برای یک ساعت ندارد
        if ($user->wallet_balance < $price_hourly_toman) {
            return response()->json([
                'status' => 'error',
                'message' => 'موجودی برای ساخت سرور کافی نیست.'
            ], 422);
        }

        // ساخت VM
        $payload = [
            'location_id'   => $data['location_id'],
            'instance_size' => $data['instance_size'],
            'template'      => $data['template'],
            'hostname'      => $data['hostname'] ?? ('oneprovider-' . uniqid()),
        ];

        $response = $this->callApi('POST', '/vm/create', $payload);
        $result = $response->json();

        if ($response->successful() && ($result['result'] ?? '') === 'success') {
            // اطلاعات اولیه VM ساخته‌شده (نیاز به GET /vm/info هم هست گاهی)
            $vm = $result['response'] ?? [];

            // ذخیره در جدول servers
            $serverData = [
                'user_id'     => $user->id,
                'project_id'  => $projectId,
                'name'        => $payload['hostname'],
                'region'      => $data['location_id'],
                'plan'        => $data['instance_size'],
                'image'       => $data['template'],
                'status'      => 'creating',
                'provider'    => 'oneprovider',
                'provider_id' => $vm['vm_id'] ?? null,
                'ip_address'  => $vm['ip_addr'] ?? null,
                'password'    => $vm['root_password'] ?? null,
                'price_hourly_toman' => $price_hourly_toman,
            ];

            $server = Server::create($serverData);

            // کسر هزینه ساعت اول
            $user->wallet_balance -= $price_hourly_toman;
            $user->save();

            // (اختیاری) ارسال ایمیل یا ثبت تراکنش

            return response()->json([
                'status'    => 'success',
                'message'   => 'سرور OneProvider با موفقیت ساخته شد.',
                'server_id' => $server->id,
            ]);
        }

        $message = $result['message'] ?? ($result['error'] ?? 'خطا در ساخت سرور!');
        return response()->json(['status' => 'error', 'message' => $message, 'response' => $result], $response->status());
    }

    // وضعیت سرور (برای نمایش و مانیتورینگ)
    public function status($serverId)
    {
        $server = Server::findOrFail($serverId);
        if ($server->provider !== 'oneprovider') {
            return response()->json(['status' => 'unknown']);
        }

        $response = $this->callApi('GET', "/server/info/{$server->provider_id}");
        $info = $response->json()['response'] ?? null;

        if ($info && isset($info['status'])) {
            $server->status = strtolower($info['status']);
            if (isset($info['ip_addr'])) $server->ip_address = $info['ip_addr'];
            $server->save();
            return response()->json([
                'status'     => $server->status,
                'ip_address' => $server->ip_address,
            ]);
        }

        return response()->json(['status' => 'unknown']);
    }

    // ریبوت سرور
    public function reboot($serverId)
    {
        $server = Server::findOrFail($serverId);
        if ($server->provider !== 'oneprovider') {
            return response()->json(['status' => 'error', 'message' => 'سرور پیدا نشد!'], 404);
        }

        // دریافت اکشن‌های مجاز
        $manageResp = $this->callApi('GET', "/server/manage/{$server->provider_id}");
        $actions = $manageResp->json()['response'] ?? [];
        if (!in_array('reboot', $actions)) {
            return response()->json(['status' => 'error', 'message' => 'عملیات ریبوت برای این سرور مجاز نیست!'], 422);
        }

        $result = $this->callApi('POST', '/server/action', [
            'server_id' => $server->provider_id,
            'action'    => 'reboot'
        ])->json();

        if (($result['result'] ?? '') === 'success') {
            $server->status = 'rebooting';
            $server->save();
            return response()->json(['status' => 'success', 'message' => 'سرور در حال ریبوت است.']);
        }
        return response()->json(['status' => 'error', 'message' => 'خطا در ریبوت سرور!']);
    }

    // خاموش/روشن کردن سرور (power)
    public function power(Request $request, $serverId)
    {
        $server = Server::findOrFail($serverId);
        if ($server->provider !== 'oneprovider') {
            return response()->json(['status' => 'error', 'message' => 'سرور پیدا نشد!'], 404);
        }

        $state = $request->input('state'); // 0=خاموش, 1=روشن

        // دریافت اکشن‌های مجاز
        $manageResp = $this->callApi('GET', "/server/manage/{$server->provider_id}");
        $actions = $manageResp->json()['response'] ?? [];

        $action = $state ? 'poweron' : 'poweroff';
        if (!in_array($action, $actions)) {
            return response()->json(['status' => 'error', 'message' => 'امکان انجام این عملیات وجود ندارد!'], 422);
        }

        $result = $this->callApi('POST', '/server/action', [
            'server_id' => $server->provider_id,
            'action'    => $action
        ])->json();

        if (($result['result'] ?? '') === 'success') {
            $server->status = $state ? 'starting' : 'stopped';
            $server->save();
            return response()->json(['status' => 'success', 'message' => 'عملیات با موفقیت انجام شد.']);
        }
        return response()->json(['status' => 'error', 'message' => 'خطا در عملیات پاور سرور!']);
    }

    // حذف سرور
    public function destroy($serverId)
    {
        $server = Server::findOrFail($serverId);
        if ($server->provider !== 'oneprovider') {
            return response()->json(['status' => 'error', 'message' => 'سرور پیدا نشد!'], 404);
        }

        // در OneProvider حذف VM روند خاص دارد، اینجا نمونه:
        $result = $this->callApi('POST', '/server/cancel', [
            'server_id' => $server->provider_id
        ])->json();

        if (($result['result'] ?? '') === 'success') {
            $server->status = 'deleting';
            $server->save();
            return response()->json(['status' => 'success', 'message' => 'درخواست حذف سرور ثبت شد.']);
        }
        return response()->json(['status' => 'error', 'message' => 'خطا در حذف سرور!']);
    }
}