<?php

namespace App\Http\Controllers;

use App\Models\Project;
use App\Models\ServerIp;
use App\Models\Server;
use App\Services\IpManagementService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class PrimaryIpController extends Controller
{
    protected $ipService;

    public function __construct(IpManagementService $ipService)
    {
        $this->ipService = $ipService;
    }

/**
 * صفحه اصلی Primary IPs (فقط مربوط به این پروژه)
 */
public function index(Project $project)
{
    // ✅ بررسی دسترسی
    if ($project->user_id !== auth()->id()) {
        abort(403, 'شما دسترسی به این پروژه ندارید');
    }

    $user = auth()->user();

    // ✅ دریافت Primary IPs که یا به سرورهای این پروژه متصل هستند یا Detached هستند
    $primaryIps = ServerIp::where('user_id', $user->id)
        ->where('is_primary', 1)
        ->whereNull('deleted_at')
        ->where(function($query) use ($project) {
            // IP های متصل به سرورهای این پروژه
            $query->whereHas('server', function($q) use ($project) {
                $q->where('project_id', $project->id);
            })
            // یا IP های Detached (که به هر سروری متصل نیستن)
            ->orWhere(function($q) {
                $q->where('status', 'detached')
                  ->whereNull('server_id');
            });
        })
        ->with(['server.project'])
        ->orderByRaw("CASE WHEN status = 'active' THEN 1 WHEN status = 'detached' THEN 2 ELSE 3 END")
        ->orderBy('created_at', 'desc')
        ->get();

    // ✅ دریافت فقط سرورهای این پروژه (Hetzner و Vultr)
    $servers = $project->servers()
        ->whereIn('provider', ['hetzner', 'vultr'])
        ->whereIn('status', ['active', 'running', 'stopped', 'off'])
        ->select('id', 'name', 'provider', 'region', 'status', 'project_id', 'ip_address')
        ->get();

    // ✅ محاسبه آمار (فقط برای این پروژه)
    $stats = [
        'total' => $primaryIps->count(),
        'assigned' => $primaryIps->where('status', 'active')->count(),
        'unassigned' => $primaryIps->where('status', 'detached')->count(),
        'ipv4_count' => $primaryIps->where('type', 'ipv4')->count(),
        'ipv6_count' => $primaryIps->where('type', 'ipv6')->count(),
        'total_hourly_cost' => $primaryIps->sum('hourly_cost'),
        'hetzner_count' => $primaryIps->where('provider', 'hetzner')->count(),
        'vultr_count' => $primaryIps->where('provider', 'vultr')->count(),
    ];

    // ✅ Location های Hetzner
    $hetznerLocations = [
        'nbg1' => 'Nuremberg (NBG1)',
        'fsn1' => 'Falkenstein (FSN1)',
        'hel1' => 'Helsinki (HEL1)',
        'ash' => 'Ashburn (ASH)',
        'hil' => 'Hillsboro (HIL)',
    ];

    return view('projects.servers.primaryips.index', compact(
        'project',
        'primaryIps',
        'servers',
        'stats',
        'hetznerLocations'
    ));
}

/**
 * ✅ فعال‌سازی IPv6 برای Vultr
 */
public function enableVultrIPv6(Request $request, Project $project, Server $server)
{
    // بررسی دسترسی
    if ($project->user_id !== auth()->id() || $server->user_id !== auth()->id()) {
        return response()->json(['success' => false, 'message' => 'دسترسی غیرمجاز'], 403);
    }

    // چک Provider
    if ($server->provider !== 'vultr') {
        return response()->json([
            'success' => false,
            'message' => 'این قابلیت فقط برای Vultr است.'
        ], 400);
    }

    // چک اینکه قبلاً IPv6 داره یا نه
    $hasIPv6 = ServerIp::where('server_id', $server->id)
        ->where('type', 'ipv6')
        ->where('is_primary', 1)
        ->exists();

    if ($hasIPv6) {
        return response()->json([
            'success' => false,
            'message' => 'این سرور قبلاً IPv6 دارد.'
        ], 400);
    }

    try {
        Log::info('🔵 Enabling IPv6 for Vultr', [
            'server_id' => $server->id,
            'provider_id' => $server->provider_id,
        ]);

        // ارسال درخواست به Vultr API
        $response = \Http::withHeaders([
            'Authorization' => 'Bearer ' . config('services.vultr.api_key'),
            'Content-Type' => 'application/json',
        ])->patch("https://api.vultr.com/v2/instances/{$server->provider_id}", [
            'enable_ipv6' => true
        ]);

        if (!$response->successful()) {
            throw new \Exception($response->json()['error'] ?? 'خطا در فعال‌سازی IPv6');
        }

        Log::info('✅ IPv6 Enable Request Sent', [
            'server_id' => $server->id,
        ]);

        // صبر 2 ثانیه و دریافت اطلاعات جدید
        sleep(2);

        // دریافت اطلاعات سرور
        $instanceResponse = \Http::withHeaders([
            'Authorization' => 'Bearer ' . config('services.vultr.api_key'),
        ])->get("https://api.vultr.com/v2/instances/{$server->provider_id}");

        if ($instanceResponse->successful()) {
            $instance = $instanceResponse->json()['instance'] ?? [];
            $v6MainIp = $instance['v6_main_ip'] ?? null;
            $v6Network = $instance['v6_network'] ?? null;

            if ($v6MainIp && $v6MainIp !== '') {
                // ذخیره IPv6 در دیتابیس
                $ipRecord = ServerIp::create([
                    'server_id' => $server->id,
                    'user_id' => $server->user_id,
                    'ip' => $v6MainIp,
                    'type' => 'ipv6',
                    'is_primary' => 1,
                    'is_floating' => 0,
                    'assigned_at' => now(),
                    'status' => 'active',
                    'provider' => 'vultr',
                    'location' => $server->region,
                    'hourly_cost' => 0,
                    'netmask' => $v6Network,
                ]);

                Log::info('✅ IPv6 Saved to Database', [
                    'ip_id' => $ipRecord->id,
                    'ip' => $v6MainIp,
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'IPv6 با موفقیت فعال شد.',
                    'ipv6' => $v6MainIp,
                ]);
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'درخواست فعال‌سازی IPv6 ارسال شد. لطفاً چند دقیقه صبر کنید.',
        ]);

    } catch (\Exception $e) {
        Log::error('❌ Enable IPv6 Failed', [
            'server_id' => $server->id,
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
        ]);

        return response()->json([
            'success' => false,
            'message' => 'خطا در فعال‌سازی IPv6: ' . $e->getMessage(),
        ], 500);
    }
}




    /**
     * ✅ ایجاد Primary IP جدید
     */
public function store(Request $request, Project $project)
{
    // بررسی دسترسی
    if ($project->user_id !== auth()->id()) {
        return response()->json([
            'success' => false,
            'message' => 'دسترسی غیرمجاز'
        ], 403);
    }

    // ولیدیشن
    $validated = $request->validate([
        'type' => 'required|in:ipv4,ipv6',
        'location' => 'required|string',
        'name' => 'nullable|string|max:255',
    ]);

    try {
        Log::info('Creating Primary IP', [
            'user_id' => auth()->id(),
            'project_id' => $project->id,
            'data' => $validated,
        ]);

        // ✅ استفاده از متد موجود
        $primaryIp = $this->ipService->createStandalonePrimaryIp(
            auth()->user(),
            $validated['type'],
            $validated['location'],
            $validated['name'] ?? null
        );

        Log::info('Primary IP created successfully', [
            'ip_id' => $primaryIp->id,
            'ip_address' => $primaryIp->ip,
            'status' => $primaryIp->status,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Primary IP با موفقیت ساخته شد',
            'ip' => [
                'id' => $primaryIp->id,
                'ip' => $primaryIp->ip,
                'type' => $primaryIp->type,
                'location' => $primaryIp->location,
                'hourly_cost' => number_format($primaryIp->hourly_cost),
                'status' => $primaryIp->status,
            ],
        ]);

    } catch (\Exception $e) {
        Log::error('Failed to create Primary IP', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
        ]);

        return response()->json([
            'success' => false,
            'message' => $e->getMessage(),
        ], 500);
    }
}

/**
 * دریافت لیست سرورهای قابل Assign برای یک IP
 */
public function getAvailableServers(Request $request, $projectId, $primaryIpId)
{
    Log::info('🔍 getAvailableServers called', [
        'user_id' => auth()->id(),
        'user_login' => auth()->user()->login ?? auth()->user()->email,
        'project_id' => $projectId,
        'primary_ip_id' => $primaryIpId,
    ]);

    try {
        $project = Project::findOrFail($projectId);
        $primaryIp = ServerIp::findOrFail($primaryIpId);

        Log::info('✅ Found project and IP', [
            'project_owner' => $project->user_id,
            'ip_owner' => $primaryIp->user_id,
            'current_user' => auth()->id(),
        ]);

        // بررسی دسترسی
        if ($project->user_id !== auth()->id()) {
            Log::warning('❌ Project ownership failed', [
                'project_owner' => $project->user_id,
                'current_user' => auth()->id(),
            ]);
            return response()->json(['success' => false, 'message' => 'پروژه متعلق به شما نیست'], 403);
        }

        if ($primaryIp->user_id !== auth()->id()) {
            Log::warning('❌ IP ownership failed', [
                'ip_owner' => $primaryIp->user_id,
                'current_user' => auth()->id(),
            ]);
            return response()->json(['success' => false, 'message' => 'IP متعلق به شما نیست'], 403);
        }

        Log::info('✅ Access granted, fetching servers', [
            'ip_location' => $primaryIp->location,
            'ip_type' => $primaryIp->type,
        ]);

        // فقط سرورهای هتزنر با همون Location
        $servers = Server::where('user_id', auth()->id())
            ->where('provider', 'hetzner')
            ->where('region', $primaryIp->location)
            ->get()
            ->map(function ($server) use ($primaryIp) {
                // چک کن این سرور از قبل این نوع IP رو داره یا نه
                $hasThisTypeIp = ServerIp::where('server_id', $server->id)
                    ->where('type', $primaryIp->type)
                    ->where('is_primary', 1)
                    ->whereIn('status', ['active', 'assigned'])
                    ->exists();

                return [
                    'id' => $server->id,
                    'name' => $server->name,
                    'status' => $server->status,
                    'region' => $server->region,
                    'ip_address' => $server->ip_address,
                    'disabled' => $hasThisTypeIp,
                    'disabled_reason' => $hasThisTypeIp 
                        ? "این سرور قبلاً یک Primary {$primaryIp->type} دارد" 
                        : null,
                ];
            });

        Log::info('✅ Found servers', [
            'count' => $servers->count(),
        ]);

        return response()->json([
            'success' => true,
            'servers' => $servers->values(),
            'ip_location' => $primaryIp->location,
            'ip_type' => $primaryIp->type,
        ]);

    } catch (\Exception $e) {
        Log::error('❌ Error in getAvailableServers', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
        ]);

        return response()->json([
            'success' => false,
            'message' => 'خطا: ' . $e->getMessage(),
        ], 500);
    }
}

public function assign(Request $request, Project $project, ServerIp $primaryIp)
{
    // بررسی دسترسی
    if ($project->user_id !== auth()->id() || $primaryIp->user_id !== auth()->id()) {
        return response()->json(['success' => false, 'message' => 'دسترسی غیرمجاز'], 403);
    }

    $validated = $request->validate([
        'server_id' => 'required|exists:servers,id',
    ]);

    try {
        $server = Server::findOrFail($validated['server_id']);

        // چک مالکیت سرور
        if ($server->user_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'این سرور متعلق به شما نیست',
            ], 403);
        }

        // ✅ چک Location
        if ($server->region !== $primaryIp->location) {
            return response()->json([
                'success' => false,
                'message' => "این IP فقط در Location {$primaryIp->location} قابل استفاده است، ولی سرور در {$server->region} است",
            ], 400);
        }

        // چک وضعیت سرور
        if (!in_array($server->status, ['stopped', 'off'])) {
            return response()->json([
                'success' => false,
                'message' => 'سرور باید خاموش باشد. لطفاً ابتدا سرور را خاموش کنید.',
            ], 400);
        }

        // ✅ چک duplicate IP
        $existingIp = ServerIp::where('server_id', $server->id)
            ->where('type', $primaryIp->type)
            ->where('is_primary', 1)
            ->whereIn('status', ['active', 'assigned'])
            ->first();

        if ($existingIp) {
            return response()->json([
                'success' => false,
                'message' => "این سرور قبلاً یک Primary {$primaryIp->type} دارد. هر سرور فقط می‌تواند یک Primary IP از هر نوع داشته باشد.",
            ], 400);
        }

        // Assign
        $this->ipService->reassignDetachedIp($server, $primaryIp);

        return response()->json([
            'success' => true,
            'message' => "IP با موفقیت به سرور {$server->name} متصل شد",
        ]);

    } catch (\Exception $e) {
        Log::error('Failed to assign Primary IP', [
            'error' => $e->getMessage(),
            'ip_id' => $primaryIp->id,
            'server_id' => $validated['server_id'] ?? null,
        ]);

        return response()->json([
            'success' => false,
            'message' => $e->getMessage(),
        ], 500);
    }
}

/**
 * Unassign از سرور
 */
public function unassign(Project $project, ServerIp $primaryIp)
{
    // بررسی دسترسی
    if ($project->user_id !== auth()->id() || $primaryIp->user_id !== auth()->id()) {
        return response()->json(['success' => false, 'message' => 'دسترسی غیرمجاز'], 403);
    }

    try {
        $server = $primaryIp->server;

        if (!$server || $server->user_id !== auth()->id()) {
            return response()->json([
                'success' => false,
                'message' => 'دسترسی غیرمجاز',
            ], 403);
        }

        if (!in_array($server->status, ['stopped', 'off'])) {
            return response()->json([
                'success' => false,
                'message' => 'سرور باید خاموش باشد. لطفاً ابتدا سرور را خاموش کنید.',
            ], 400);
        }

        $this->ipService->unassignPrimaryIp($primaryIp);

        return response()->json([
            'success' => true,
            'message' => 'IP با موفقیت از سرور جدا شد',
        ]);

    } catch (\Exception $e) {
        Log::error('Failed to unassign Primary IP', [
            'error' => $e->getMessage(),
            'ip_id' => $primaryIp->id,
        ]);

        return response()->json([
            'success' => false,
            'message' => $e->getMessage(),
        ], 500);
    }
}
    /**
     * حذف دائمی
     */
    public function destroy(Project $project, ServerIp $primaryIp)
    {
        // بررسی دسترسی
        if ($project->user_id !== auth()->id() || $primaryIp->user_id !== auth()->id()) {
            return response()->json(['success' => false, 'message' => 'دسترسی غیرمجاز'], 403);
        }

        try {
            $this->ipService->deletePermanentlyDetachedIp($primaryIp);

            return response()->json([
                'success' => true,
                'message' => 'Primary IP با موفقیت حذف شد',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }

/**
 * آپدیت Reverse DNS (Hetzner & Vultr)
 */
public function updateRdns(Request $request, Project $project, ServerIp $primaryIp)
{
    // بررسی دسترسی
    if ($project->user_id !== auth()->id() || $primaryIp->user_id !== auth()->id()) {
        return response()->json(['success' => false, 'message' => 'دسترسی غیرمجاز'], 403);
    }

    $validated = $request->validate([
        'rdns' => 'required|string|max:255|regex:/^[a-zA-Z0-9\.\-]+$/',
    ], [
        'rdns.required' => 'لطفاً Reverse DNS را وارد کنید',
        'rdns.regex' => 'فرمت Reverse DNS نامعتبر است',
    ]);

    try {
        $reverse = $validated['rdns'];
        $server = $primaryIp->server;

        if (!$server) {
            throw new \Exception('سرور مرتبط یافت نشد');
        }

        Log::info('🔵 Updating RDNS', [
            'ip_id' => $primaryIp->id,
            'ip' => $primaryIp->ip,
            'provider' => $primaryIp->provider,
            'reverse' => $reverse,
        ]);

        // ✅ تشخیص Provider و اجرای API مربوطه
        if ($primaryIp->provider === 'hetzner') {
            // Hetzner API
            $response = \Http::withHeaders([
                'Authorization' => 'Bearer ' . config('services.hetzner.api_key'),
                'Content-Type' => 'application/json',
            ])->post("https://api.hetzner.cloud/v1/primary_ips/actions/change_dns_ptr", [
                'ip' => $primaryIp->ip,
                'dns_ptr' => $reverse,
            ]);

            if (!$response->successful()) {
                $error = $response->json()['error']['message'] ?? 'خطا در تنظیم RDNS';
                throw new \Exception($error);
            }

        } elseif ($primaryIp->provider === 'vultr') {
            // Vultr API
            $response = \Http::withHeaders([
                'Authorization' => 'Bearer ' . config('services.vultr.api_key'),
                'Content-Type' => 'application/json',
            ])->post("https://api.vultr.com/v2/instances/{$server->provider_id}/ipv4/reverse", [
                'ip' => $primaryIp->ip,
                'reverse' => $reverse,
            ]);

            if (!$response->successful()) {
                $error = $response->json()['error'] ?? 'خطا در تنظیم RDNS';
                throw new \Exception($error);
            }

        } else {
            throw new \Exception('Provider پشتیبانی نمی‌شود');
        }

        // به‌روزرسانی دیتابیس
        $primaryIp->update([
            'rdns' => $reverse,
            'reverse_dns' => $reverse,
        ]);

        Log::info('✅ RDNS Updated Successfully', [
            'ip_id' => $primaryIp->id,
            'provider' => $primaryIp->provider,
            'reverse' => $reverse,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Reverse DNS با موفقیت به‌روزرسانی شد',
            'rdns' => $reverse,
        ]);

    } catch (\Exception $e) {
        Log::error('❌ RDNS Update Failed', [
            'ip_id' => $primaryIp->id,
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
        ]);

        return response()->json([
            'success' => false,
            'message' => 'خطا در به‌روزرسانی RDNS: ' . $e->getMessage(),
        ], 500);
    }
  }
}