<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Models\Server;
use App\Models\User;
use App\Mail\ServerCreatedMail;
use Illuminate\Support\Facades\Mail;
use App\Services\LimitService;
use App\Services\WalletService;

class VultrController extends Controller
{
    private $apiKey;

    // نرخ ثابت دلار به تومان
    private $dollarRate = 120000;

    public function __construct(LimitService $limitService, WalletService $walletService)
    {
        $this->limitService = $limitService;
        $this->walletService = $walletService;
        $this->apiKey = config('services.vultr.api_key');
    }

    // لیست دیتاسنترها (Regions)
    public function regions()
    {
        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->apiKey,
            'Accept'        => 'application/json',
        ])->get('https://api.vultr.com/v2/regions');

        if ($response->successful()) {
            $data = $response->json();
            $regions = collect($data['regions'] ?? [])->values()->all();
            return response()->json([
                'status' => 'success',
                'regions' => $regions,
            ]);
        }
        return response()->json(['status' => 'error', 'message' => 'خطا در دریافت regions از Vultr'], $response->status());
    }

// لیست پلن‌ها بر اساس region (فقط پلن‌های فعال همان دیتاسنتر)
public function plansByRegion($region)
{
    $plansResponse = Http::withHeaders([
        'Authorization' => 'Bearer ' . $this->apiKey,
        'Accept'        => 'application/json',
    ])->get('https://api.vultr.com/v2/plans?type=vc2');

    $availResponse = Http::withHeaders([
        'Authorization' => 'Bearer ' . $this->apiKey,
        'Accept'        => 'application/json',
    ])->get("https://api.vultr.com/v2/regions/{$region}/availability");

    if ($plansResponse->successful() && $availResponse->successful()) {
        $plansData = $plansResponse->json();
        $availData = $availResponse->json();

        $plans = $plansData['plans'] ?? [];
        $availableIds = $availData['plans'] ?? [];

        if (empty($availableIds)) {
            foreach ($availData as $key => $val) {
                if (is_array($val)) {
                    $availableIds = $val;
                    break;
                }
            }
        }

        $filteredPlans = collect($plans)->filter(function($plan) use ($availableIds) {
            $planId = $plan['id'] ?? '';
            
            // شرط 1: پلن در این region موجود باشه
            $isAvailable = in_array($planId, $availableIds);
            
            // شرط 2: پلن IPv6-only نباشه (حذف پلن‌های -v6)
            $isNotIPv6Only = !str_ends_with($planId, '-v6');
            
            return $isAvailable && $isNotIPv6Only; // ✅ این خط اصلاح شده
        })->values()->all();

        return response()->json([
            'status' => 'success',
            'plans' => $filteredPlans,
        ]);
    }
    
    return response()->json(['status' => 'error', 'message' => 'خطا در دریافت پلن Vultr']);
}

    // لیست ایمیج‌ها (OS) فقط x64
    public function images()
    {
        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->apiKey,
            'Accept'        => 'application/json',
        ])->get('https://api.vultr.com/v2/os');

        if ($response->successful()) {
            $data = $response->json();
            $images = collect($data['os'] ?? [])->filter(function ($item) {
                return !isset($item['arch']) || $item['arch'] === 'x64';
            })->values()->all();

            return response()->json([
                'status' => 'success',
                'images' => $images,
            ]);
        }
        return response()->json(['status' => 'error', 'message' => 'خطا در دریافت ایمیج Vultr'], $response->status());
    }

    // ساخت سرور جدید Vultr و ذخیره در جدول servers
public function createVM(Request $request, $projectId)
{
    \Log::info('VULTR-createVM: درخواست دریافتی', [
        'project_id' => $projectId,
        'all' => $request->all(),
    ]);

    $data = $request->validate([
        'region' => 'required|string',
        'plan' => 'required|string',
        'os_id' => 'required|integer',
        'label' => 'required|string',
    ]);

    \Log::info('VULTR-createVM: داده validate شده', $data);

    $user = User::find(auth()->id());
    $min_balance = 50000;

    \Log::info('CHECK_BALANCE_VULTR', [
        'wallet_balance' => $user->wallet_balance ?? 0,
        'min_balance' => $min_balance
    ]);

    if (!$user || $user->wallet_balance < $min_balance) {
        return response()->json([
            'status' => 'error',
            'message' => 'موجودی کیف پول کافی نیست.'
        ], 422);
    }
    
       // ✅ چک محدودیت سرورها
        if (!$this->limitService->canCreateServer($user->id)) {
            $info = $this->limitService->getLimitInfo($user->id);
            return response()->json([
                'status' => 'error',
                'message' => 'شما به حد محدودیت سرورهای مجاز رسیده‌اید',
                'limit_info' => $info['servers'],
                'current_servers' => $info['servers']['used'],
                'max_servers' => $info['servers']['limit']
            ], 403);
        }

    // دریافت قیمت پلن
    $plansResponse = Http::withHeaders([
        'Authorization' => 'Bearer ' . $this->apiKey,
        'Accept' => 'application/json',
    ])->get('https://api.vultr.com/v2/plans?type=vc2');

    $plans = $plansResponse->json()['plans'] ?? [];
    $selectedPlan = collect($plans)->firstWhere('id', $data['plan']);

    if (!$selectedPlan) {
        return response()->json([
            'status' => 'error',
            'message' => 'پلن انتخابی یافت نشد.'
        ], 404);
    }

    $priceHourlyUSD = $selectedPlan['hourly_cost'] ?? 0;
    $dollarRate = 120000;
    $priceHourlyToman = $priceHourlyUSD * $dollarRate;
    
    // ✅ قیمت کل (شامل Primary IPv4) با 30% سود
    $finalHourlyPrice = ceil($priceHourlyToman * 1.30);

    \Log::info('VULTR-createVM: محاسبه هزینه ساعتی', [
        'plan_id' => $data['plan'],
        'price_hourly_usd' => $priceHourlyUSD,
        'dollar_rate' => $dollarRate,
        'price_hourly_toman' => $priceHourlyToman,
        'final_hourly_price' => $finalHourlyPrice,
        'note' => 'شامل Primary IPv4 رایگان',
    ]);

    // ساخت سرور
    $payload = [
        'region' => $data['region'],
        'plan' => $data['plan'],
        'os_id' => $data['os_id'],
        'label' => $data['label'],
    ];

    \Log::info('VULTR-createVM: payload ارسالی به Vultr', $payload);

    $response = Http::withHeaders([
        'Authorization' => 'Bearer ' . $this->apiKey,
        'Content-Type' => 'application/json',
    ])->post('https://api.vultr.com/v2/instances', $payload);

    \Log::info('VULTR-createVM: پاسخ Vultr', [
        'status' => $response->status(),
        'body' => $response->body(),
    ]);

    if ($response->successful()) {
        $responseData = $response->json();
        $instance = $responseData['instance'] ?? [];

        \Log::info('VULTR-createVM: instance دریافتی', $instance);

        // دریافت IP ها
        $mainIp = $instance['main_ip'] ?? null;
        $v6MainIp = $instance['v6_main_ip'] ?? null;
        
        // اگه هنوز IP نگرفته، 0.0.0.0 میاد
        if ($mainIp === '0.0.0.0' || $mainIp === '') {
            $mainIp = null;
        }
        
        if ($v6MainIp === '' || $v6MainIp === '::') {
            $v6MainIp = null;
        }

        // ✅ ذخیره سرور (قیمت شامل Primary IPv4)
        $serverData = [
            'user_id' => auth()->id(),
            'project_id' => $projectId,
            'name' => $instance['label'] ?? $data['label'],
            'region' => $data['region'],
            'plan' => $data['plan'],
            'image' => (string) $data['os_id'],
            'status' => 'creating',
            'provider' => 'vultr',
            'provider_id' => $instance['id'] ?? null,
            'ip_address' => $mainIp,
            'password' => $instance['default_password'] ?? null,
            'price_hourly_toman' => $finalHourlyPrice, // ✅ شامل Primary IPv4
        ];

        \Log::info('Server Data Before Create', $serverData);

        $server = Server::create($serverData);

        \Log::info('Server After Create', $server->toArray());

// ✅ ذخیره Primary IPv4 (با مشخصات شبکه)
if ($mainIp) {
    $ipRecord = \App\Models\ServerIp::create([
        'server_id' => $server->id,
        'user_id' => $server->user_id,
        'ip' => $mainIp,
        'type' => 'ipv4',
        'is_primary' => 1,
        'is_floating' => 0,
        'assigned_at' => now(),
        'status' => 'active',
        'provider' => 'vultr',
        'location' => $data['region'],
        'hourly_cost' => 0, // رایگان (شامل پلن)
        'netmask' => $instance['netmask_v4'] ?? null, // ✅ جدید
        'gateway' => $instance['gateway_v4'] ?? null, // ✅ جدید
        'reverse_dns' => null, // ✅ هنوز ست نشده (بعداً با sync)
    ]);

    \Log::info('✅ Vultr Primary IPv4 saved', [
        'server_id' => $server->id,
        'ip_record_id' => $ipRecord->id,
        'ip' => $mainIp,
        'netmask' => $instance['netmask_v4'] ?? null,
        'gateway' => $instance['gateway_v4'] ?? null,
    ]);
}

// ✅ ذخیره Primary IPv6 (با مشخصات شبکه)
if ($v6MainIp) {
    $ipv6Record = \App\Models\ServerIp::create([
        'server_id' => $server->id,
        'user_id' => $server->user_id,
        'ip' => $v6MainIp,
        'type' => 'ipv6',
        'is_primary' => 1,
        'is_floating' => 0,
        'assigned_at' => now(),
        'status' => 'active',
        'provider' => 'vultr',
        'location' => $data['region'],
        'hourly_cost' => 0,
        'netmask' => $instance['v6_network'] ?? null, // ✅ IPv6 Network
        'gateway' => null, // IPv6 معمولاً gateway جدا نداره
        'reverse_dns' => null,
    ]);

    \Log::info('✅ Vultr Primary IPv6 saved', [
        'server_id' => $server->id,
        'ip_record_id' => $ipv6Record->id,
        'ip' => $v6MainIp,
        'v6_network' => $instance['v6_network'] ?? null,
    ]);
}


// 💰 کسر هزینه ساعت اول سرور (شامل Primary IPv4)
try {
    $walletService = app(\App\Services\WalletService::class);
    
    // کسر هزینه ساعت اول
    // توجه: چون Primary IPv4 رایگان هست و شامل پلن میشه،
    // فقط یک بار کل هزینه رو کسر می‌کنیم
    $serverTransaction = $walletService->deductServerHourlyCost(
        $user,
        $server->id,
        $server->name,
        $finalHourlyPrice, // کل هزینه شامل IPv4
        'vultr'
    );

    if ($serverTransaction) {
        \Log::info('💰 First hour cost deducted (Vultr)', [
            'server_id' => $server->id,
            'amount' => $finalHourlyPrice,
            'transaction_id' => $serverTransaction->id,
            'note' => 'شامل Primary IPv4 رایگان',
        ]);
    } else {
        \Log::warning('⚠️ Failed to deduct first hour cost - insufficient balance', [
            'server_id' => $server->id,
            'required_amount' => $finalHourlyPrice,
            'user_balance' => $user->wallet_balance,
        ]);
        
        // اختیاری: می‌تونید سرور رو حذف کنید یا status رو به suspended تغییر بدید
        // $server->update(['status' => 'suspended']);
    }
    
} catch (\Exception $e) {
    \Log::error('❌ Error deducting first hour cost (Vultr)', [
        'server_id' => $server->id,
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString(),
    ]);
}


// ارسال ایمیل
try {
    $userEmail = $user->email;
    \Log::info('TRY_SEND_EMAIL_QUEUE', [
        'user_email' => $userEmail,
        'server_id' => $server->id,
        'panel_url' => config('app.url') . '/cloud/',
    ]);

    // ✅ اصلاح: پاس دادن user, server, panel_url
    \Mail::to($userEmail)->later(
        now()->addSeconds(30),
        new \App\Mail\ServerCreatedMail(
            $user,                              // ← User object
            $server,                            // ← Server object
            config('app.url') . '/cloud/'       // ← Panel URL
        )
    );

    \Log::info('EMAIL_QUEUED', [
        'user_email' => $userEmail,
        'server_id' => $server->id,
        'send_after' => '30 seconds',
    ]);
} catch (\Exception $e) {
    \Log::error('EMAIL_SEND_FAILED', [
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString(),
        'server_id' => $server->id,
    ]);
}

        return response()->json([
            'status' => 'success',
            'message' => 'سرور Vultr با موفقیت ساخته شد.',
            'server_id' => $server->id,
        ]);
    }

    \Log::error('VULTR-createVM: خطا در ساخت', [
        'status' => $response->status(),
        'body' => $response->body(),
    ]);

    return response()->json([
        'status' => 'error',
        'message' => 'خطا در ساخت سرور Vultr!',
        'error' => $response->json(),
    ], $response->status());
}

    // نمایش اطلاعات سرور Vultr (overview)
    public function overview($serverId)
    {
        $dbServer = Server::findOrFail($serverId);
        $apiData = null;
        $password = null;

        if ($dbServer->provider === 'vultr' && $dbServer->provider_id) {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Accept'        => 'application/json',
            ])->get("https://api.vultr.com/v2/instances/{$dbServer->provider_id}");

            if ($response->successful()) {
                $apiData = (object)($response->json()['instance'] ?? []);
                $password = $apiData->default_password ?? null;
            }
        }

        // لیست سیستم عامل‌ها (برای نصب مجدد)
        $osList = [];
        $osResponse = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->apiKey,
            'Accept'        => 'application/json',
        ])->get('https://api.vultr.com/v2/os');

        if ($osResponse->successful()) {
            $allOs = $osResponse->json()['os'] ?? [];
            $osList = [];
            foreach ($allOs as $os) {
                // اگر شیء بود، به آرایه تبدیل کن
                if (is_object($os)) $os = (array)$os;
                if (!in_array($os['family'], ['iso','snapshot','backup','application'])) {
                    $osList[] = $os;
                }
            }
        }
        if (!is_array($osList)) $osList = []; // تضمین نوع آرایه

        return view('servers.overview', [
            'server' => $dbServer,
            'apiData' => $apiData ?? null,
            'password' => $password ?? null,
            'osList' => $osList, // مهم: همین نام
        ]);
    }

    // ریبوت سرور Vultr
    public function reboot($serverId)
    {
        Log::info('VULTR-reboot: درخواست', [
            'server_id' => $serverId
        ]);
        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->apiKey,
            'Content-Type'  => 'application/json',
            'Accept'        => 'application/json',
        ])->post("https://api.vultr.com/v2/instances/{$serverId}/reboot");

        Log::info('VULTR-reboot: پاسخ Vultr', [
            'status' => $response->status(),
            'body' => $response->body(),
        ]);

        return response()->json([
            'status' => $response->successful() ? 'success' : 'error',
            'message' => $response->successful() ? 'سرور با موفقیت ریستارت شد.' : 'خطا در ریستارت!',
            'data' => $response->json()
        ], $response->status());
    }

    // نصب مجدد سیستم عامل Vultr
    public function reinstall(Request $request, $serverId)
    {
        $request->validate([
            'os_id' => 'required|numeric',
        ]);

        Log::info('VULTR-reinstall: درخواست', [
            'server_id' => $serverId,
            'os_id' => $request->os_id
        ]);

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->apiKey,
            'Content-Type'  => 'application/json',
            'Accept'        => 'application/json',
        ])->post("https://api.vultr.com/v2/instances/{$serverId}/reinstall", [
            'os_id' => $request->os_id,
        ]);

        Log::info('VULTR-reinstall: پاسخ Vultr', [
            'status' => $response->status(),
            'body' => $response->body(),
        ]);

        if ($response->successful()) {
            // رمز جدید را اگر وجود داشت ذخیره کن
            $data = $response->json();
            if (!empty($data['instance']['default_password'])) {
                $server = Server::where('provider', 'vultr')->where('provider_id', $serverId)->first();
                if ($server) {
                    $server->password = $data['instance']['default_password'];
                    $server->save();
                    Log::info('VULTR-reinstall: رمز جدید ذخیره شد', ['password' => $server->password]);
                }
            }
            return response()->json(['success' => true, 'message' => 'نصب مجدد آغاز شد']);
        }
        return response()->json(['success' => false, 'message' => 'خطا در نصب مجدد!'], 500);
    }

    // تغییر وضعیت پاور سرور Vultr (روشن/خاموش)
public function power(Request $request, $serverId)
{
    $state = $request->input('state'); // 0=خاموش, 1=روشن
    $endpoint = $state ? 'start' : 'halt';

    // پیدا کردن سرور در دیتابیس
    $server = Server::where('provider', 'vultr')->where('provider_id', $serverId)->first();
    if (!$server) {
        return response()->json([
            'status' => 'error',
            'message' => 'سرور یافت نشد!'
        ], 404);
    }

    // ممنوعیت روشن کردن سرور ساسپند یا کم‌اعتبار
    $user = $server->user;
    if ($state == 1 && ($server->status === 'suspended' || ($user && $user->wallet_balance < 5000))) {
        return response()->json([
            'status' => 'error',
            'message' => 'امکان روشن کردن این سرور وجود ندارد (مسدود یا اعتبار ناکافی).'
        ], 403);
    }

    Log::info('VULTR-power: درخواست', [
        'server_id' => $serverId,
        'state' => $state,
        'endpoint' => $endpoint,
    ]);

    $response = Http::withHeaders([
        'Authorization' => 'Bearer ' . $this->apiKey,
        'Content-Type'  => 'application/json',
        'Accept'        => 'application/json',
    ])->post("https://api.vultr.com/v2/instances/{$serverId}/{$endpoint}");

    Log::info('VULTR-power: پاسخ Vultr', [
        'status' => $response->status(),
        'body' => $response->body(),
    ]);

    return response()->json([
        'status' => $response->successful() ? 'success' : 'error',
        'message' => $response->successful() ? 'عملیات با موفقیت انجام شد.' : 'خطا در عملیات!',
        'data' => $response->json()
    ], $response->status());
}

    // حذف سرور Vultr
    public function destroy($serverId)
    {
        Log::info('VULTR-destroy: درخواست حذف', [
            'server_id' => $serverId
        ]);
        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->apiKey,
            'Accept'        => 'application/json',
        ])->delete("https://api.vultr.com/v2/instances/{$serverId}");

        Log::info('VULTR-destroy: پاسخ Vultr', [
            'status' => $response->status(),
            'body' => $response->body(),
        ]);

        // در صورت حذف موفق، اگر رکورد دیتابیس وجود داشت حذف کن (برای یکپارچگی جدول servers)
        $server = Server::where('provider', 'vultr')->where('provider_id', $serverId)->first();
        if ($response->successful() && $server) {
            Log::info('VULTR-destroy: حذف رکورد دیتابیس', ['server_row' => $server->toArray()]);
            $server->delete();
        }

        return response()->json([
            'status' => $response->successful() ? 'success' : 'error',
            'message' => $response->successful() ? 'سرور با موفقیت حذف شد.' : 'خطا در حذف!',
            'data' => $response->json()
        ], $response->status());
    }
}