<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\DB;
use App\Models\Invoice;
use App\Models\User;
use App\Models\Server;
use App\Models\WalletTransaction;
use App\Mail\InvoiceCreatedMail;
use App\Mail\InvoicePaidMail;
use Illuminate\Support\Facades\Mail;
use App\Services\WalletService;
use App\Models\GiftCode;
use App\Models\GiftCodeUsage;


class WalletController extends Controller
{
    protected $walletService;

    public function __construct(WalletService $walletService)
    {
        $this->walletService = $walletService;
    }

    /**
     * نمایش صفحه اصلی کیف پول (با تب‌ها)
     */
public function showChargeForm(Request $request)  // ✅ اضافه شد
{
    $user = Auth::user();
    $wallet_balance = $user->wallet_balance ?? 0;
    
    $invoices = Invoice::where('user_id', $user->id)
        ->orderBy('id', 'desc')
        ->limit(10)
        ->get();
    
    // دریافت تب فعال از query string
    $activeTab = $request->get('tab', 'charge');
    
    return view('wallet-charge', compact('wallet_balance', 'invoices', 'activeTab'));
}

/**
 * اعمال کد هدیه
 */
public function applyGiftCode(Request $request)
{
    $request->validate([
        'code' => 'required|string|max:50',
    ]);

    $user = Auth::user();
    $code = strtoupper(trim($request->code));

    try {
        // جستجو کد
        $giftCode = GiftCode::where('code', $code)->first();

        if (!$giftCode) {
            return response()->json([
                'status' => 'error',
                'message' => 'کد هدیه معتبر نیست!'
            ], 422);
        }

        // چک اعتبار کد
        if (!$giftCode->isValid()) {
            $message = 'این کد هدیه ';
            
            if (!$giftCode->is_active) {
                $message .= 'غیرفعال است!';
            } elseif ($giftCode->used_count >= $giftCode->max_uses) {
                $message .= 'تمام شده است!';
            } else {
                $message .= 'منقضی شده است!';
            }
            
            return response()->json([
                'status' => 'error',
                'message' => $message
            ], 422);
        }

        // چک استفاده قبلی
        if (!$giftCode->canBeUsedBy($user->id)) {
            return response()->json([
                'status' => 'error',
                'message' => 'شما قبلاً از این کد هدیه استفاده کرده‌اید!'
            ], 422);
        }

        // اجرای تراکنش - بدون DB::beginTransaction
        // چون WalletService::deposit خودش transaction داره
        $walletService = app(\App\Services\WalletService::class);
        
        $transaction = $walletService->deposit(
            $user,
            $giftCode->amount,
            "استفاده از کد هدیه: {$giftCode->code}",
            null,
            [
                'gift_code_id' => $giftCode->id,
                'gift_code' => $giftCode->code,
                'applied_at' => now(),
            ]
        );

        // ثبت استفاده (بدون transaction)
        GiftCodeUsage::create([
            'user_id' => $user->id,
            'gift_code_id' => $giftCode->id,
            'amount' => $giftCode->amount,
            'used_at' => now(),
        ]);

        // افزایش شمارنده
        $giftCode->increment('used_count');

        // بروز کردن موجودی
        $user->refresh();

        Log::info('🎁 Gift code applied', [
            'user_id' => $user->id,
            'code' => $giftCode->code,
            'amount' => $giftCode->amount,
            'transaction_id' => $transaction->id ?? null,
        ]);

        return response()->json([
            'status' => 'success',
            'message' => '🎉 تبریک! ' . number_format($giftCode->amount) . ' تومان به حساب شما اضافه شد.',
            'amount' => $giftCode->amount,
            'new_balance' => $user->wallet_balance,
        ]);

    } catch (\Exception $e) {
        Log::error('Gift code error', [
            'user_id' => $user->id ?? null,
            'code' => $code ?? null,
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
        ]);

        return response()->json([
            'status' => 'error',
            'message' => 'خطا در اعمال کد هدیه. لطفاً دوباره تلاش کنید.'
        ], 500);
    }
}


    /**
     * پردازش درخواست شارژ و ساخت فاکتور
     */
    public function charge(Request $request)
    {
        $request->validate([
            'amount' => 'required|integer|min:20000|max:5000000',
            'method' => 'required|in:bank'
        ], [
            'amount.required' => 'لطفاً مبلغ شارژ را وارد کنید',
            'amount.min' => 'حداقل مبلغ شارژ 20,000 تومان است',
            'amount.max' => 'حداکثر مبلغ شارژ 5,000,000 تومان است',
        ]);

        $invoice = Invoice::create([
            'user_id' => Auth::id(),
            'amount' => $request->amount,
            'method' => $request->method,
            'status' => 'pending'
        ]);

        $user = Auth::user();
        
        // ارسال ایمیل فاکتور جدید
        if ($user && $user->email) {
            try {
                Mail::to($user->email)->send(new InvoiceCreatedMail($user, $invoice));
                \Log::info('EMAIL_SENT: InvoiceCreatedMail', [
                    'user_id' => $user->id,
                    'invoice_id' => $invoice->id
                ]);
            } catch (\Exception $e) {
                \Log::error('EMAIL_ERROR: InvoiceCreatedMail', [
                    'user_email' => $user->email,
                    'invoice_id' => $invoice->id,
                    'error' => $e->getMessage()
                ]);
            }
        }

        // هدایت به درگاه پرداخت
        if ($request->method === 'bank') {
            return redirect()->route('wallet.zibal.pay', $invoice->id);
        }

        return redirect()->route('wallet.charge.form')
            ->with('danger', 'روش پرداخت نامعتبر است.');
    }

    /**
     * اتصال به درگاه زیبال
     */
    public function zibalPay($invoice_id)
    {
        $invoice = Invoice::where('id', $invoice_id)
            ->where('user_id', Auth::id())
            ->where('status', 'pending')
            ->firstOrFail();

        $merchant = env('ZIBAL_MERCHANT', 'xxxxxxx');
        $amount = $invoice->amount * 10; // تبدیل تومان به ریال
        $callbackUrl = route('wallet.zibal.callback', $invoice->id);

        $response = Http::post('https://gateway.zibal.ir/v1/request', [
            'merchant' => $merchant,
            'amount' => $amount,
            'callbackUrl' => $callbackUrl,
            'orderId' => $invoice->id,
            'mobile' => Auth::user()->phone ?? null,
        ]);

        $result = $response->json();

        if (isset($result['result']) && $result['result'] == 100) {
            return redirect('https://gateway.zibal.ir/start/' . $result['trackId']);
        } else {
            $msg = $result['message'] ?? 'خطای نامشخص در پرداخت';
            \Log::error('Zibal Request Error', [
                'invoice_id' => $invoice_id,
                'response' => $result
            ]);
            return back()->with('danger', 'خطا در درخواست پرداخت: ' . $msg);
        }
    }

    /**
     * Callback از درگاه زیبال
     */
    public function zibalCallback(Request $request, $invoice_id)
    {
        $invoice = Invoice::where('id', $invoice_id)->firstOrFail();
        $merchant = env('ZIBAL_MERCHANT', 'xxxxxxx');
        $trackId = $request->input('trackId');
        $success = $request->input('success');

        // چک کردن success از query string
        if ($success != 1) {
            $invoice->status = 'unpaid';
            $invoice->save();
            
            return redirect()->route('wallet.charge.form')
                ->with('danger', 'پرداخت توسط کاربر لغو شد');
        }

        // تأیید پرداخت
        $response = Http::post('https://gateway.zibal.ir/v1/verify', [
            'merchant' => $merchant,
            'trackId' => $trackId,
        ]);
        
        $result = $response->json();

        if (isset($result['result']) && $result['result'] == 100) {
            // پرداخت موفق
            $invoice->status = 'paid';
            $invoice->transaction_id = $trackId;
            $invoice->save();

            $user = User::find($invoice->user_id);
            
            // استفاده از WalletService برای افزودن موجودی
            $transaction = $this->walletService->deposit(
                $user,
                $invoice->amount,
                "شارژ کیف پول از طریق درگاه بانکی - فاکتور #{$invoice->id}",
                $invoice->id,
                [
                    'payment_method' => 'zibal',
                    'track_id' => $trackId,
                    'invoice_id' => $invoice->id,
                    'paid_at' => now()->toDateTimeString(),
                ]
            );

            \Log::info('Wallet Charged Successfully', [
                'user_id' => $user->id,
                'invoice_id' => $invoice->id,
                'amount' => $invoice->amount,
                'transaction_id' => $transaction->id,
            ]);

            // ارسال ایمیل تأیید پرداخت
            if ($user->email) {
                try {
                    Mail::to($user->email)->send(new InvoicePaidMail($user, $invoice));
                    \Log::info('EMAIL_SENT: InvoicePaidMail', [
                        'user_id' => $user->id,
                        'invoice_id' => $invoice->id
                    ]);
                } catch (\Exception $e) {
                    \Log::error('EMAIL_ERROR: InvoicePaidMail', [
                        'user_email' => $user->email,
                        'invoice_id' => $invoice->id,
                        'error' => $e->getMessage()
                    ]);
                }
            }

            return redirect()->route('wallet.charge.form')
                ->with('success', '✅ پرداخت با موفقیت انجام شد و ' . number_format($invoice->amount) . ' تومان به کیف پول شما افزوده شد.');
        }

        // پرداخت ناموفق
        $invoice->status = 'unpaid';
        $invoice->save();
        
        $msg = $result['message'] ?? 'پرداخت ناموفق';
        \Log::warning('Payment Failed', [
            'invoice_id' => $invoice_id,
            'track_id' => $trackId,
            'result' => $result
        ]);
        
        return redirect()->route('wallet.charge.form')
            ->with('danger', '❌ پرداخت ناموفق: ' . $msg);
    }
    
    
    /**
 * دریافت هزینه‌های یک سرور خاص
 */
public function getServerCosts(Request $request, Server $server)
{
    $user = Auth::user();
    
    // بررسی مالکیت
    if ($server->user_id !== $user->id) {
        abort(403, 'شما به این سرور دسترسی ندارید');
    }
    
    $serverId = $server->id;
    
    $from = $request->get('from', now()->startOfMonth()->toDateString());
    $to = $request->get('to', now()->toDateString());
    $page = $request->get('page', 1);
    $perPage = 20;
    
    // دریافت تراکنش‌ها و آمار
    $transactions = $this->walletService->getServerTransactions($serverId, $from, $to, $perPage);
    $stats = $this->walletService->getServerCostStats($serverId, $from, $to);
    
    // فرمت کردن تراکنش‌ها
    $formattedTransactions = $transactions->map(function($tx) {
        return [
            'id' => $tx->id,
            'type' => $tx->type,
            'type_label' => $tx->getTypeLabel(),
            'type_icon' => $tx->getTypeIcon(),
            'description' => $tx->description,
            'amount' => $tx->amount,
            'amount_formatted' => number_format(abs($tx->amount)) . ' تومان',
            'status' => $tx->status,
            'created_at' => $tx->created_at->format('Y/m/d H:i'),
            'created_at_human' => $tx->created_at->diffForHumans(),
            'metadata' => $tx->metadata,
        ];
    });
    
    return response()->json([
        'success' => true,
        'server' => [
            'id' => $server->id,
            'name' => $server->name,
        ],
        'transactions' => $formattedTransactions,
        'stats' => [
            'total_cost' => $stats['total_cost'],
            'total_cost_formatted' => number_format($stats['total_cost']) . ' تومان',
            'server_costs' => $stats['server_costs'],
            'server_costs_formatted' => number_format($stats['server_costs']) . ' تومان',
            'ip_costs' => $stats['ip_costs'],
            'ip_costs_formatted' => number_format($stats['ip_costs']) . ' تومان',
            'transaction_count' => $stats['transaction_count'],
            'period' => $stats['period'],
        ],
        'pagination' => [
            'current_page' => $transactions->currentPage(),
            'last_page' => $transactions->lastPage(),
            'per_page' => $transactions->perPage(),
            'total' => $transactions->total(),
            'from' => $transactions->firstItem(),
            'to' => $transactions->lastItem(),
        ]
    ]);
}
    

    /**
     * نمایش صفحه تراکنش‌ها (برای مسیر جداگانه)
     */
public function transactions(Request $request)
{
    $user = Auth::user();
    
    $type = $request->get('type');
    $from = $request->get('from', now()->startOfMonth()->toDateString());
    $to = $request->get('to', now()->toDateString());
    
    // اگر درخواست AJAX باشه، JSON برگردون
    if ($request->ajax() || $request->wantsJson()) {
        return $this->getTransactionsJson($user, $type, $from, $to, $request->get('page', 1));
    }
    
    // در غیر این صورت به صفحه اصلی با تب transactions
    return redirect()->route('wallet.charge.form', [
        'tab' => 'transactions'
    ] + $request->all()); // + تمام query params
}

    /**
     * دریافت تراکنش‌ها به صورت JSON (برای AJAX)
     */
    public function getTransactions(Request $request)
    {
        $user = Auth::user();
        
        $type = $request->get('type');
        $from = $request->get('from', now()->startOfMonth()->toDateString());
        $to = $request->get('to', now()->toDateString());
        $page = $request->get('page', 1);
        
        return $this->getTransactionsJson($user, $type, $from, $to, $page);
    }

    /**
     * متد کمکی برای ساخت JSON تراکنش‌ها
     */
/**
 * متد کمکی برای ساخت JSON تراکنش‌ها
 */
private function getTransactionsJson($user, $type, $from, $to, $page)
{
    $perPage = 20;
    $transactions = $this->walletService->getUserTransactions($user, $type, $from, $to, $perPage);
    $stats = $this->walletService->getUserStats($user, $from, $to);
    
    // فرمت کردن تراکنش‌ها برای JSON
    $formattedTransactions = $transactions->map(function($tx) {
        return [
            'id' => $tx->id,
            'type' => $tx->type,
            'type_label' => $tx->getTypeLabel(),
            'type_icon' => $tx->getTypeIcon(),
            'description' => $tx->description,
            'amount' => $tx->amount,
            'amount_formatted' => number_format(abs($tx->amount)) . ' تومان',
            'balance_before' => $tx->balance_before ?? 0,
            'balance_before_formatted' => number_format($tx->balance_before ?? 0) . ' تومان',
            'balance_after' => $tx->balance_after ?? 0,
            'balance_after_formatted' => number_format($tx->balance_after ?? 0) . ' تومان',
            'status' => $tx->status,
            'status_label' => $tx->getStatusLabel(),
            'status_class' => $tx->getStatusClass(),
            'created_at' => $tx->created_at->format('Y/m/d H:i'),
            'created_at_human' => $tx->created_at->diffForHumans(),
            'is_deposit' => $tx->isDeposit(),
        ];
    });
    
    return response()->json([
        'success' => true,
        'transactions' => $formattedTransactions,
        'stats' => [
            'current_balance' => $stats['current_balance'] ?? 0,
            'current_balance_formatted' => number_format($stats['current_balance'] ?? 0) . ' تومان',
            'total_deposits' => $stats['total_deposits'] ?? 0,
            'total_deposits_formatted' => number_format($stats['total_deposits'] ?? 0) . ' تومان',
            'total_costs' => $stats['total_costs'] ?? 0,
            'total_costs_formatted' => number_format($stats['total_costs'] ?? 0) . ' تومان',
            'server_costs' => $stats['server_costs'] ?? 0,
            'server_costs_formatted' => number_format($stats['server_costs'] ?? 0) . ' تومان',
            'ip_costs' => $stats['ip_costs'] ?? 0,
            'ip_costs_formatted' => number_format($stats['ip_costs'] ?? 0) . ' تومان',
            'transaction_count' => $stats['transaction_count'] ?? 0,
        ],
        'pagination' => [
            'current_page' => $transactions->currentPage(),
            'last_page' => $transactions->lastPage(),
            'per_page' => $transactions->perPage(),
            'total' => $transactions->total(),
            'from' => $transactions->firstItem(),
            'to' => $transactions->lastItem(),
        ]
    ]);
}

    /**
     * نمایش جزئیات فاکتور
     */
    public function showInvoice(Invoice $invoice)
    {
        // بررسی مالکیت
        if ($invoice->user_id !== Auth::id()) {
            abort(403, 'شما به این فاکتور دسترسی ندارید');
        }
        
        return view('wallet-invoice', compact('invoice'));
    }

    /**
     * پرداخت فاکتور معلق
     */
    public function payInvoice(Invoice $invoice)
    {
        // بررسی مالکیت
        if ($invoice->user_id !== Auth::id()) {
            abort(403, 'شما به این فاکتور دسترسی ندارید');
        }
        
        if ($invoice->status !== 'pending') {
            return redirect()->route('wallet.charge.form')
                ->with('warning', 'این فاکتور قبلاً پرداخت شده یا منقضی شده است');
        }
        
        // هدایت به درگاه بانک
        return redirect()->route('wallet.zibal.pay', $invoice->id);
    }

    /**
     * دانلود فاکتور (PDF - اختیاری)
     */
    public function downloadInvoice(Invoice $invoice)
    {
        if ($invoice->user_id !== Auth::id()) {
            abort(403, 'دسترسی غیرمجاز');
        }
        
        // اینجا می‌تونید PDF بسازید
        // فعلاً صفحه چاپ نمایش میدیم
        return view('wallet-invoice-print', compact('invoice'));
    }
}