<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class WhmcsSsoController extends Controller
{
    private $whmcsSecret;

    public function __construct()
    {
        $this->whmcsSecret = config('services.whmcs.secret');
    }

    /**
     * دریافت و ثبت کاربر جدید از WHMCS
     */
    public function createUser(Request $request)
    {
        // بررسی امنیت
        if (!$this->verifyWhmcsRequest($request)) {
            return response()->json(['error' => 'Unauthorized'], 401);
        }

        $validated = $request->validate([
            'whmcs_id' => 'required|integer',
            'firstname' => 'required|string',
            'lastname' => 'required|string',
            'email' => 'required|email|unique:users,email',
            'phonenumber' => 'nullable|string',
            'country' => 'nullable|string',
            'city' => 'nullable|string',
            'state' => 'nullable|string',
            'postcode' => 'nullable|string',
            'address1' => 'nullable|string',
        ]);

        try {
            $user = User::create([
                'whmcs_id' => $validated['whmcs_id'],
                'name' => $validated['firstname'] . ' ' . $validated['lastname'],
                'email' => $validated['email'],
                'password' => Hash::make(Str::random(32)), // پسورد رندوم
                'phone' => $validated['phonenumber'] ?? null,
                'country' => $validated['country'] ?? null,
                'city' => $validated['city'] ?? null,
                'state' => $validated['state'] ?? null,
                'postal_code' => $validated['postcode'] ?? null,
                'address' => $validated['address1'] ?? null,
                'email_verified_at' => now(), // ایمیل از قبل در WHMCS تایید شده
            ]);

            return response()->json([
                'success' => true,
                'user_id' => $user->id,
                'message' => 'User created successfully'
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to create user',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * آپدیت اطلاعات کاربر
     */
    public function updateUser(Request $request)
    {
        if (!$this->verifyWhmcsRequest($request)) {
            return response()->json(['error' => 'Unauthorized'], 401);
        }

        $validated = $request->validate([
            'whmcs_id' => 'required|integer',
            'firstname' => 'required|string',
            'lastname' => 'required|string',
            'email' => 'required|email',
            'phonenumber' => 'nullable|string',
            'country' => 'nullable|string',
            'city' => 'nullable|string',
            'state' => 'nullable|string',
            'postcode' => 'nullable|string',
            'address1' => 'nullable|string',
        ]);

        try {
            $user = User::where('whmcs_id', $validated['whmcs_id'])->first();

            if (!$user) {
                // اگر کاربر وجود نداره، ایجادش کن
                return $this->createUser($request);
            }

            $user->update([
                'name' => $validated['firstname'] . ' ' . $validated['lastname'],
                'email' => $validated['email'],
                'phone' => $validated['phonenumber'] ?? null,
                'country' => $validated['country'] ?? null,
                'city' => $validated['city'] ?? null,
                'state' => $validated['state'] ?? null,
                'postal_code' => $validated['postcode'] ?? null,
                'address' => $validated['address1'] ?? null,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'User updated successfully'
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to update user',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * SSO Login - ورود کاربر با توکن
     */
/**
 * SSO Login - ورود کاربر با توکن
 */
public function ssoLogin(Request $request)
{
    $token = $request->query('token');

    \Log::info('--- SSO Login Debug Start ---');
    \Log::info('Received Token: ' . $token);

    if (!$token) {
        \Log::warning('No token provided');
        return redirect('/login')->with('error', 'توکن معتبر نیست');
    }

    // تجزیه توکن
    $parts = explode('.', $token);
    if (count($parts) !== 2) {
        \Log::warning('Token format invalid', ['parts_count' => count($parts)]);
        return redirect('/login')->with('error', 'فرمت توکن نامعتبر است');
    }

    [$payloadEncoded, $signature] = $parts;

    // اصلاح Base64 برای decoding
    $payloadForDecode = strtr($payloadEncoded, '-_', '+/');
    $payloadForDecode .= str_repeat('=', 3 - (strlen($payloadForDecode) + 3) % 4);

    // محاسبه signature مورد انتظار روی Base64 اصلی
    $expectedSignature = hash_hmac('sha256', $payloadEncoded, $this->whmcsSecret);

    \Log::info('Payload (encoded): ' . $payloadEncoded);
    \Log::info('Signature (from token): ' . $signature);
    \Log::info('Expected Signature: ' . $expectedSignature);
    \Log::info('Signature length: ' . strlen($signature));
    \Log::info('Expected signature length: ' . strlen($expectedSignature));

    // decode payload
    $decodedPayload = base64_decode($payloadForDecode, true);
    \Log::info('Payload (decoded): ' . $decodedPayload);

    $data = json_decode($decodedPayload, true);
    \Log::info('Payload (json): ', $data ?? []);

    // بررسی امضا
    if (!hash_equals($expectedSignature, $signature)) {
        \Log::error('Signature mismatch! Login aborted.');
        return redirect('/login')->with('error', 'توکن معتبر نیست');
    }

    // بررسی انقضا
    if ($data['exp'] < time()) {
        \Log::warning('Token expired', ['exp' => $data['exp'], 'now' => time()]);
        return redirect('/login')->with('error', 'توکن منقضی شده است');
    }

    // پیدا کردن کاربر
    $user = User::where('whmcs_id', $data['whmcs_id'])
                ->orWhere('email', $data['email'])
                ->first();

    if (!$user) {
        $this->fetchUserFromWhmcs($data['whmcs_id']);
        $user = User::where('whmcs_id', $data['whmcs_id'])->first();
    }

    if (!$user) {
        \Log::error('User not found in database', ['whmcs_id' => $data['whmcs_id']]);
        return redirect('/login')->with('error', 'کاربر یافت نشد');
    }

    // لاگین
    Auth::login($user, true);
    \Log::info('User logged in successfully', ['user_id' => $user->id, 'email' => $user->email]);

    \Log::info('--- SSO Login Debug End ---');

    return redirect('/cloud')->with('success', 'با موفقیت وارد شدید');
}


    /**
     * دریافت اطلاعات کاربر از WHMCS
     */
    private function fetchUserFromWhmcs($whmcsId)
    {
        \Log::info("fetchUserFromWhmcs called for WHMCS ID: $whmcsId");
        
        // مثال ساده:
        try {
            $whmcsUrl = config('services.whmcs.url');
            $whmcsUsername = config('services.whmcs.username');
            $whmcsPassword = config('services.whmcs.password');
  \Log::info("Connecting to WHMCS API at $whmcsUrl with username $whmcsUsername");
            $postData = [
                'action' => 'GetClientsDetails',
                'clientid' => $whmcsId,
                'username' => $whmcsUsername,
                'password' => $whmcsPassword,
                'responsetype' => 'json',
            ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $whmcsUrl . '/includes/api.php');
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($postData));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false); // اگر SSL مشکل داشت
        $response = curl_exec($ch);
        $curlError = curl_error($ch);
        curl_close($ch);

        if ($curlError) {
            \Log::error("Curl error while fetching WHMCS user: $curlError");
            return;
        }

        $result = json_decode($response, true);

        if (!$result) {
            \Log::error("Failed to decode WHMCS response: $response");
            return;
        }

        \Log::info("WHMCS API response: " . json_encode($result));

        if ($result['result'] === 'success') {
            try {
                $user = User::create([
                    'whmcs_id' => $whmcsId,
                    'name' => ($result['firstname'] ?? 'Unknown') . ' ' . ($result['lastname'] ?? 'User'),
                    'email' => $result['email'] ?? "user{$whmcsId}@example.com",
                    'password' => Hash::make(Str::random(32)),
                    'phone' => $result['phonenumber'] ?? null,
                    'country' => $result['country'] ?? null,
                    'city' => $result['city'] ?? null,
                    'state' => $result['state'] ?? null,
                    'postal_code' => $result['postcode'] ?? null,
                    'address' => $result['address1'] ?? null,
                    'email_verified_at' => now(),
                ]);

                \Log::info('User created successfully', ['user_id' => $user->id, 'whmcs_id' => $whmcsId]);

            } catch (\Exception $e) {
                \Log::error('Failed to create user in database: ' . $e->getMessage());
            }
        } else {
            \Log::warning('WHMCS API returned error', ['message' => $result['message'] ?? 'Unknown error']);
        }

    } catch (\Exception $e) {
        \Log::error('Failed to fetch user from WHMCS: ' . $e->getMessage());
    }
}

    /**
     * بررسی امنیت درخواست از WHMCS
     */
    private function verifyWhmcsRequest(Request $request)
    {
        $secret = $request->header('X-WHMCS-Secret');
        return hash_equals($this->whmcsSecret, $secret);
    }
}