<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;

class ServerIp extends Model
{
    protected $fillable = [
        'server_id',
        'user_id',
        'ip',
        'type',
        'rdns',
        'is_primary',
        'is_floating',
        'assigned_at',
        'detached_at',
        'suspended_at',
        'last_charged_at', // ✅ اضافه شد
        'provider_ip_id',
        'provider',
        'datacenter',
        'status',
        'location',
        'hourly_cost',
    ];

    /**
     * ✅ تبدیل فیلدها به Carbon
     */
    protected $casts = [
        'is_primary' => 'boolean',
        'is_floating' => 'boolean',
        'assigned_at' => 'datetime',
        'detached_at' => 'datetime',
        'suspended_at' => 'datetime', // ✅ اصلاح شد
        'last_charged_at' => 'datetime', // ✅ اضافه شد
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
        'hourly_cost' => 'decimal:2',
    ];

    // ==================== هزینه‌های ثابت ====================
    
    // Hetzner Primary IPs (ساعتی - جدا از سرور)
    const HETZNER_PRIMARY_IPV4_HOURLY = 250;  // تومان/ساعت
    const HETZNER_PRIMARY_IPV6_HOURLY = 0;    // ✅ رایگان
    
    // Hetzner Floating IPs (ماهانه - بدون جریمه) ✅
    const HETZNER_FLOATING_IPV4_MONTHLY = 750000;   // تومان/ماه
    const HETZNER_FLOATING_IPV6_MONTHLY = 300000;   // تومان/ماه
    
    // هزینه تغییر Primary IP ✅
    const IP_CHANGE_FEE = 3700; // تومان
    const IP_CHANGE_MONTHLY_LIMIT = 5; // تعداد مجاز در ماه
    
    // Vultr
    const VULTR_IPV4_FLOATING_HOURLY = 480; // تومان/ساعت

    // ==================== Relations ====================

    public function server()
    {
        return $this->belongsTo(Server::class);
    }
    
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    // ==================== Scopes ====================

    public function scopePrimary($query)
    {
        return $query->where('is_primary', 1);
    }

    public function scopeFloating($query)
    {
        return $query->where('is_floating', 1);
    }

    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopeDetached($query)
    {
        return $query->where('status', 'detached');
    }

    public function scopeSuspended($query)
    {
        return $query->where('status', 'suspended');
    }

    // ==================== Helper Methods ====================

    public function isPrimary()
    {
        return $this->is_primary == 1;
    }

    public function isFloating()
    {
        return $this->is_floating == 1;
    }
    
    public function isDetached(): bool
    {
        return $this->status === 'detached';
    }

    public function isSuspended(): bool
    {
        return $this->status === 'suspended';
    }

    public function isDeletable(): bool
    {
        return $this->isFloating() && in_array($this->status, ['active', 'assigned']);
    }
    
    public function getTypeLabel(): string
    {
        return strtoupper($this->type);
    }

    public function isIpv6()
    {
        return strtolower($this->type) === 'ipv6';
    }

    public function getProvider()
    {
        return $this->provider ?? ($this->server ? $this->server->provider : null);
    }

    /**
     * دریافت هزینه ساعتی IP
     */
    public function getHourlyPrice()
    {
        $provider = $this->getProvider();

        // Hetzner Primary IPs (ساعتی)
        if ($provider === 'hetzner' && $this->isPrimary()) {
            if ($this->isIpv6()) {
                return self::HETZNER_PRIMARY_IPV6_HOURLY; // 0 (رایگان)
            }
            return self::HETZNER_PRIMARY_IPV4_HOURLY; // 250
        }

        // Hetzner Floating IPs (ساعتی ندارند) ✅
        if ($provider === 'hetzner' && $this->isFloating()) {
            return 0;
        }

        // Vultr (ساعتی)
        if ($provider === 'vultr') {
            return self::VULTR_IPV4_FLOATING_HOURLY;
        }

        return 0;
    }
    
    /**
     * محاسبه هزینه از زمان Detach تا الان
     */
    public function calculateDetachedCost(): float
    {
        if (!$this->isDetached() || !$this->detached_at) {
            return 0;
        }

        $hoursSinceDetach = $this->detached_at->diffInHours(now());
        return $hoursSinceDetach * ($this->hourly_cost ?? 0);
    }

    /**
     * تعداد ساعت از زمان Detach
     */
    public function getHoursSinceDetach(): int
    {
        if (!$this->detached_at) {
            return 0;
        }

        return (int) $this->detached_at->diffInHours(now());
    }

    /**
     * تعداد ساعت از زمان معلق شدن
     */
    public function getHoursSinceSuspension(): int
    {
        if (!$this->suspended_at) {
            return 0;
        }

        return (int) $this->suspended_at->diffInHours(now());
    }

    /**
     * تعداد ساعت از زمان ساخت
     */
    public function getHoursSinceCreation(): int
    {
        return (int) $this->created_at->diffInHours(now());
    }

    /**
     * آیا IP باید حذف بشه؟ (بعد از 3 ساعت معلق و 48 ساعت از ساخت)
     */
    public function shouldBeDeleted(): bool
    {
        return $this->isSuspended() 
            && $this->getHoursSinceSuspension() >= 3
            && $this->getHoursSinceCreation() >= 48;
    }

    /**
     * آیا IP قابل حذف دستی است؟ (حداقل 48 ساعت از ساخت)
     */
    public function canBeDeletedManually(): bool
    {
        return $this->getHoursSinceCreation() >= 48 && $this->isDetached();
    }

    /**
     * دریافت هزینه ماهانه IP
     */
    public function getMonthlyPrice()
    {
        $provider = $this->getProvider();

        // Hetzner Primary IPs (محاسبه از ساعتی)
        if ($provider === 'hetzner' && $this->isPrimary()) {
            return $this->getHourlyPrice() * 730;
            // IPv4: 250 * 730 = 182,500 تومان/ماه
            // IPv6: 0 (رایگان)
        }

        // Hetzner Floating IPs (ثابت ماهانه) ✅
        if ($provider === 'hetzner' && $this->isFloating()) {
            if ($this->isIpv6()) {
                return self::HETZNER_FLOATING_IPV6_MONTHLY; // 300,000
            }
            return self::HETZNER_FLOATING_IPV4_MONTHLY; // 750,000
        }

        // Vultr (محاسبه از ساعتی)
        if ($provider === 'vultr') {
            return $this->getHourlyPrice() * 730;
        }

        return 0;
    }

    /**
     * محاسبه جریمه حذف
     * ✅ Floating IPs جریمه ندارند (چون ماهانه از اول کسر میشه)
     */
    public function calculateDeletionPenalty()
    {
        // هیچ IP ای جریمه حذف ندارد ✅
        return 0;
    }

    public function hasDeletionPenalty()
    {
        return false; // ✅ هیچوقت جریمه نداریم
    }

    public function getDaysActive()
    {
        if (!$this->assigned_at) {
            return 0;
        }
        $endDate = $this->detached_at ?? now();
        return $this->assigned_at->diffInDays($endDate);
    }

    public function getHoursActive()
    {
        if (!$this->assigned_at) {
            return 0;
        }
        $endDate = $this->detached_at ?? now();
        return $this->assigned_at->diffInHours($endDate);
    }

    /**
     * دریافت توضیحات هزینه
     */
    public function getCostDescription()
    {
        $hourly = $this->getHourlyPrice();
        $monthly = $this->getMonthlyPrice();

        if ($hourly == 0 && $monthly == 0) {
            return "رایگان";
        }

        $provider = $this->getProvider();

        // Hetzner Primary (ساعتی)
        if ($provider === 'hetzner' && $this->isPrimary()) {
            if ($hourly == 0) {
                return "رایگان (IPv6)";
            }
            return number_format($hourly) . " تومان/ساعت (~" . number_format($monthly) . " تومان/ماه)";
        }

        // Hetzner Floating (ماهانه - ثابت) ✅
        if ($provider === 'hetzner' && $this->isFloating()) {
            return number_format($monthly) . " تومان/ماه";
        }

        // Vultr (ساعتی)
        return number_format($hourly) . " تومان/ساعت (~" . number_format($monthly) . " تومان/ماه)";
    }

    /**
     * دریافت وضعیت حذف
     */
    public function getDeletionStatus()
    {
        if (!$this->isDeletable()) {
            $reasons = [];
            
            if ($this->isPrimary() && $this->getProvider() === 'vultr') {
                $reasons[] = 'Primary IP در Vultr قابل حذف نیست';
            }
            
            if ($this->isPrimary() && $this->getProvider() === 'hetzner') {
                $reasons[] = 'Primary IP در Hetzner قابل حذف نیست. از گزینه "تغییر IP" استفاده کنید';
            }

            if ($this->isFloating()) {
                $reasons[] = 'سرور باید خاموش باشد';
            }
            
            return [
                'deletable' => false,
                'reasons' => $reasons,
            ];
        }

        return [
            'deletable' => true,
            'has_penalty' => false, // ✅ هیچوقت جریمه نداریم
            'penalty_amount' => 0,
            'penalty_description' => null,
            'days_active' => $this->getDaysActive(),
        ];
    }

    /**
     * آیا این IP قابل Reassign هست؟
     */
    public function canBeReassigned(): bool
    {
        return $this->isPrimary() 
            && $this->status === 'detached' 
            && $this->getProvider() === 'hetzner';
    }

    /**
     * دریافت هزینه ساعتی IP
     */
    public function getHourlyCost(): float
    {
        return $this->hourly_cost ?? 0;
    }

    /**
     * دریافت متن وضعیت
     */
public function getStatusLabel(): string
{
    $labels = [
        'active' => 'فعال',
        'detached' => 'جدا شده',
        'assigned' => 'تخصیص یافته',
        'suspended' => 'معلق',
        'deleted' => 'حذف شده',
    ];

    return $labels[$this->status] ?? $this->status;
}

/**
 * دریافت Badge Color برای وضعیت
 */
public function getStatusBadgeColor(): string
{
    $colors = [
        'active' => 'success',
        'detached' => 'warning',
        'assigned' => 'primary',
        'suspended' => 'secondary',
        'deleted' => 'danger',
    ];

    return $colors[$this->status] ?? 'secondary';
}

    // ==================== Events ====================

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($ip) {
            if (!$ip->assigned_at) {
                $ip->assigned_at = now();
            }
        });

        static::deleting(function ($ip) {
            if (!$ip->detached_at) {
                $ip->detached_at = now();
            }
            
            Log::info('🗑️ IP Deleted', [
                'ip_id' => $ip->id,
                'ip' => $ip->ip,
                'type' => $ip->getTypeLabel(),
                'days_active' => $ip->getDaysActive(),
                'hours_since_creation' => $ip->getHoursSinceCreation(),
            ]);
        });
    }
}