<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Spatie\Permission\Traits\HasRoles;
use Spatie\MediaLibrary\InteractsWithMedia;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\MediaLibrary\HasMedia;

class User extends Authenticatable implements HasMedia
{
    use HasFactory, SoftDeletes, Notifiable, InteractsWithMedia, HasRoles;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'email',
        'password',
        'phone',
        'whmcs_uuid',
        'whmcs_id',
        'dob',
        'gender',
        'status',
        'first_name',
        'last_name',
        'postal_code',
        'country_id',
        'state_id',
        'location',
        'about_me',
        'bio',
        'skills',
        'country_code',
        'created_by_id',
        'system_reserve',
        'wallet_balance',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'phone' => 'integer',
            'status' => 'integer',
            'created_by_id' => 'integer',
            'wallet_balance' => 'decimal:2',
        ];
    }

    protected $appends = [
        'role',
    ];

    protected $with = [
        'media'
    ];

    public static function booted()
    {
        parent::boot();
        
        static::saving(function ($model) {
            $model->created_by_id = \App\Helpers\Helpers::isUserLogin() 
                ? \App\Helpers\Helpers::getCurrentUserId() 
                : $model->id;
        });

        // ایجاد wallet_balance پیش‌فرض برای کاربران جدید
        static::creating(function ($user) {
            if (!isset($user->wallet_balance)) {
                $user->wallet_balance = 0;
            }
        });
    }

    // ==================== Relations ====================

    /**
     * Get the user's role. (سازگار با PHP 7.4)
     */
    public function getRoleAttribute()
    {
        $firstRole = $this->roles->first();
        return $firstRole ? $firstRole->makeHidden(['created_at', 'updated_at', 'pivot']) : null;
    }

    /**
     * Get the user's all permissions.
     */
    public function getPermissionAttribute()
    {
        return $this->getAllPermissions();
    }

    /**
     * رابطه با کشور
     */
    public function country()
    {
        return $this->belongsTo(Country::class, 'country_id');
    }

    /**
     * رابطه با استان/ایالت
     */
    public function state()
    {
        return $this->belongsTo(State::class, 'state_id');
    }

    /**
     * رابطه قدیمی با wallet (برای سازگاری)
     */
    public function wallet()
    {
        return (object)[
            'balance' => $this->wallet_balance,
            'user_id' => $this->id,
        ];
    }

    /**
     * رابطه با سرورها
     */
    public function servers()
    {
        return $this->hasMany(\App\Models\Server::class, 'user_id');
    }

    /**
     * رابطه با فاکتورها
     */
    public function invoices()
    {
        return $this->hasMany(\App\Models\Invoice::class, 'user_id');
    }

    /**
     * نام جایگزین برای invoices (برای سازگاری)
     */
    public function walletInvoices()
    {
        return $this->invoices();
    }

    /**
     * رابطه با تراکنش‌های کیف پول
     */
    public function walletTransactions()
    {
        return $this->hasMany(\App\Models\WalletTransaction::class, 'user_id');
    }

    /**
     * دریافت آخرین تراکنش کیف پول
     */
    public function lastWalletTransaction()
    {
        return $this->hasOne(\App\Models\WalletTransaction::class, 'user_id')
            ->latest();
    }

    /**
     * تراکنش‌های واریز (Deposits)
     */
    public function deposits()
    {
        return $this->hasMany(\App\Models\WalletTransaction::class, 'user_id')
            ->where('type', 'deposit')
            ->where('status', 'completed');
    }

    /**
     * تراکنش‌های برداشت/هزینه (Withdrawals)
     */
    public function withdrawals()
    {
        return $this->hasMany(\App\Models\WalletTransaction::class, 'user_id')
            ->whereIn('type', ['server_hourly', 'ip_hourly', 'bandwidth_overage', 'penalty'])
            ->where('status', 'completed');
    }

    /**
     * پروژه‌ها (اگر دارید)
     */
    public function projects()
    {
        return $this->hasMany(\App\Models\Project::class, 'user_id');
    }

    // ==================== Scopes ====================

    /**
     * فقط کاربران فعال
     */
    public function scopeActive($query)
    {
        return $query->where('status', 1);
    }

    /**
     * فقط کاربران غیرفعال
     */
    public function scopeInactive($query)
    {
        return $query->where('status', 0);
    }

    /**
     * کاربرانی که موجودی کمتر از مقدار مشخص دارند
     */
    public function scopeLowBalance($query, $amount = 5000)
    {
        return $query->where('wallet_balance', '<', $amount);
    }

    /**
     * جستجوی کاربر
     */
    public function scopeSearch($query, $search)
    {
        return $query->where(function($q) use ($search) {
            $q->where('email', 'like', "%{$search}%")
              ->orWhere('first_name', 'like', "%{$search}%")
              ->orWhere('last_name', 'like', "%{$search}%")
              ->orWhere('phone', 'like', "%{$search}%");
        });
    }

    // ==================== Accessors & Mutators ====================

    /**
     * دریافت نام کامل
     */
    public function getFullNameAttribute()
    {
        $fullName = trim("{$this->first_name} {$this->last_name}");
        return $fullName ? $fullName : $this->email;
    }

    /**
     * دریافت موجودی با فرمت
     */
    public function getFormattedBalanceAttribute()
    {
        $balance = $this->wallet_balance ? $this->wallet_balance : 0;
        return number_format($balance) . ' تومان';
    }

    /**
     * دریافت نام کاربری (از email)
     */
    public function getUsernameAttribute()
    {
        return explode('@', $this->email)[0];
    }

    // ==================== Helper Methods ====================

    /**
     * آیا کاربر موجودی کافی دارد؟
     */
    public function hasEnoughBalance($amount)
    {
        $balance = $this->wallet_balance ? $this->wallet_balance : 0;
        return $balance >= $amount;
    }

    /**
     * آیا کاربر ادمین است؟
     */
    public function isAdmin()
    {
        return $this->hasRole('admin') || $this->hasRole('super-admin');
    }

    /**
     * آیا کاربر فعال است؟
     */
    public function isActive()
    {
        return $this->status == 1;
    }

    /**
     * دریافت تعداد سرورهای فعال
     */
    public function getActiveServersCount()
    {
        return $this->servers()
            ->whereIn('status', ['active', 'running', 'creating'])
            ->count();
    }

    /**
     * دریافت مجموع هزینه‌های ماهانه
     */
    public function getMonthlyServerCost()
    {
        return $this->servers()
            ->whereIn('status', ['active', 'running'])
            ->sum('price_hourly_toman') * 730;
    }

    /**
     * دریافت آمار کیف پول
     */
    public function getWalletStats($from = null, $to = null)
    {
        $from = $from ? $from : now()->startOfMonth();
        $to = $to ? $to : now();

        $deposits = $this->walletTransactions()
            ->where('type', 'deposit')
            ->where('status', 'completed')
            ->whereBetween('created_at', [$from, $to])
            ->sum('amount');

        $costs = abs($this->walletTransactions()
            ->whereIn('type', ['server_hourly', 'ip_hourly', 'bandwidth_overage', 'penalty'])
            ->where('status', 'completed')
            ->whereBetween('created_at', [$from, $to])
            ->sum('amount'));

        $balance = $this->wallet_balance ? $this->wallet_balance : 0;

        return [
            'current_balance' => $balance,
            'total_deposits' => $deposits,
            'total_costs' => $costs,
            'period_from' => $from,
            'period_to' => $to,
        ];
    }

    /**
     * دریافت اعلان‌های خوانده نشده
     */
    public function getUnreadNotificationsCount()
    {
        return $this->unreadNotifications()->count();
    }
    
    
    /**
 * بررسی محدودیت تغییر IP ماهانه
 */
public function canChangeIp()
{
    // ریست شمارنده اگر ماه جدید شده
    if ($this->ip_change_reset_at && $this->ip_change_reset_at->isLastMonth()) {
        $this->ip_change_count = 0;
        $this->ip_change_reset_at = now()->startOfMonth();
        $this->save();
    }

    return $this->ip_change_count < \App\Models\ServerIp::IP_CHANGE_MONTHLY_LIMIT;
}

/**
 * افزایش شمارنده تغییر IP
 */
public function incrementIpChangeCount()
{
    if (!$this->ip_change_reset_at) {
        $this->ip_change_reset_at = now()->startOfMonth();
    }

    $this->ip_change_count++;
    $this->save();
}

/**
 * دریافت تعداد تغییرات باقی‌مانده
 */
public function getRemainingIpChanges()
{
    $this->canChangeIp(); // ریست میکنه اگر لازم باشه
    return \App\Models\ServerIp::IP_CHANGE_MONTHLY_LIMIT - $this->ip_change_count;
}
    
    
}