<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;

class WalletTransaction extends Model
{
    protected $fillable = [
        'user_id',
        'amount',
        'type',
        'status',
        'description',
        'ref_type',
        'ref_id',
        'invoice_id',
        'balance_before',
        'balance_after',
        'metadata',
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'balance_before' => 'decimal:2',
        'balance_after' => 'decimal:2',
        'metadata' => 'array',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    // ==================== Relations ====================

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function invoice()
    {
        return $this->belongsTo(Invoice::class);
    }

    public function referenceable()
    {
        return $this->morphTo(__FUNCTION__, 'ref_type', 'ref_id');
    }
    
    public function serverIp()
    {
    return $this->belongsTo(\App\Models\ServerIp::class, 'ref_id');
    }

    // ==================== Scopes ====================

    /**
     * فقط واریزی‌ها
     */
    public function scopeDeposits($query)
    {
        return $query->where('type', 'deposit');
    }

    /**
     * فقط برداشت‌ها
     */
    public function scopeWithdrawals($query)
    {
        return $query->whereIn('type', [
            'server_hourly',
            'ip_hourly',
            'server_create',
            'ip_create',
            'bandwidth_overage',
            'penalty'
        ]);
    }

    /**
     * فقط تراکنش‌های تکمیل شده
     */
    public function scopeCompleted($query)
    {
        return $query->where('status', 'completed');
    }

    /**
     * فقط تراکنش‌های در انتظار
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * فیلتر بر اساس بازه زمانی ✅ این مهم بود!
     */
    public function scopeForPeriod($query, $from, $to)
    {
        return $query->whereBetween('created_at', [$from, $to]);
    }

    /**
     * فیلتر بر اساس ماه جاری
     */
    public function scopeCurrentMonth($query)
    {
        return $query->whereBetween('created_at', [
            now()->startOfMonth(),
            now()->endOfMonth()
        ]);
    }

    /**
     * فیلتر بر اساس نوع خاص
     */
    public function scopeOfType($query, $type)
    {
        return $query->where('type', $type);
    }

    /**
     * مرتب‌سازی بر اساس جدیدترین
     */
    public function scopeLatestFirst($query)
    {
        return $query->orderBy('created_at', 'desc');
    }

    // ==================== Helper Methods ====================

    /**
     * آیا این تراکنش واریز است؟
     */
    public function isDeposit()
    {
        return $this->amount > 0 || $this->type === 'deposit';
    }

    /**
     * آیا این تراکنش برداشت است؟
     */
    public function isWithdrawal()
    {
        return $this->amount < 0 || in_array($this->type, [
            'server_hourly',
            'ip_hourly',
            'server_create',
            'ip_create',
            'bandwidth_overage',
            'penalty'
        ]);
    }

    /**
     * دریافت لیبل فارسی نوع تراکنش
     */
    public function getTypeLabel()
    {
        $labels = [
            'deposit' => 'شارژ کیف پول',
            'withdrawal' => 'برداشت وجه',
            'server_hourly' => 'هزینه ساعتی سرور',
            'ip_hourly' => 'هزینه ساعتی IP',
            'server_create' => 'ساخت سرور',
            'ip_create' => 'ایجاد IP',
            'bandwidth_overage' => 'اضافه ترافیک',
            'refund' => 'بازگشت وجه',
            'bonus' => 'هدیه',
            'penalty' => 'جریمه',
        ];

        return isset($labels[$this->type]) ? $labels[$this->type] : $this->type;
    }

    /**
     * دریافت آیکون نوع تراکنش
     */
    public function getTypeIcon()
    {
        $icons = [
            'deposit' => 'ri-add-circle-line',
            'withdrawal' => 'ri-subtract-line',
            'server_hourly' => 'ri-server-line',
            'ip_hourly' => 'ri-global-line',
            'server_create' => 'ri-add-box-line',
            'ip_create' => 'ri-router-line',
            'bandwidth_overage' => 'ri-speed-line',
            'refund' => 'ri-refund-2-line',
            'bonus' => 'ri-gift-line',
            'penalty' => 'ri-alert-line',
        ];

        return isset($icons[$this->type]) ? $icons[$this->type] : 'ri-money-dollar-circle-line';
    }

    /**
     * دریافت لیبل فارسی وضعیت
     */
    public function getStatusLabel()
    {
        $labels = [
            'pending' => 'در انتظار',
            'completed' => 'تکمیل شده',
            'failed' => 'ناموفق',
            'refunded' => 'بازگشت داده شده',
        ];

        return isset($labels[$this->status]) ? $labels[$this->status] : $this->status;
    }

    /**
     * دریافت کلاس رنگ وضعیت برای Badge
     */
    public function getStatusClass()
    {
        $classes = [
            'pending' => 'warning',
            'completed' => 'success',
            'failed' => 'danger',
            'refunded' => 'info',
        ];

        return isset($classes[$this->status]) ? $classes[$this->status] : 'secondary';
    }

    /**
     * دریافت کلاس رنگ مبلغ
     */
    public function getAmountClass()
    {
        return $this->amount > 0 ? 'text-success' : 'text-danger';
    }

    /**
     * فرمت کردن مبلغ با علامت + یا -
     */
    public function getFormattedAmount()
    {
        $sign = $this->amount > 0 ? '+' : '';
        return $sign . number_format($this->amount) . ' تومان';
    }

    /**
     * دریافت لینک منبع (اگر وجود داشته باشد)
     */
    public function getSourceLink()
    {
        if (!$this->ref_type || !$this->ref_id) {
            return null;
        }

        if ($this->ref_type === 'server') {
            return route('servers.overview', $this->ref_id);
        }

        if ($this->ref_type === 'invoice') {
            return route('wallet.charge.form');
        }

        return null;
    }

    /**
     * دریافت نام منبع
     */
    public function getSourceName()
    {
        if (!$this->referenceable) {
            return null;
        }

        if ($this->ref_type === 'server') {
            $name = isset($this->referenceable->name) ? $this->referenceable->name : null;
            return $name ? $name : "سرور #{$this->ref_id}";
        }

        if ($this->ref_type === 'ip') {
            $ip = isset($this->referenceable->ip) ? $this->referenceable->ip : null;
            return $ip ? $ip : "IP #{$this->ref_id}";
        }

        if ($this->ref_type === 'invoice') {
            return "فاکتور #{$this->ref_id}";
        }

        return "#{$this->ref_id}";
    }

    // ==================== Static Methods ====================

    /**
     * محاسبه مجموع واریزی‌ها در یک بازه زمانی
     */
    public static function totalDeposits($userId, $from = null, $to = null)
    {
        $query = static::where('user_id', $userId)
            ->deposits()
            ->completed();

        if ($from && $to) {
            $query->forPeriod($from, $to);
        }

        return $query->sum('amount');
    }

    /**
     * محاسبه مجموع برداشت‌ها در یک بازه زمانی
     */
    public static function totalWithdrawals($userId, $from = null, $to = null)
    {
        $query = static::where('user_id', $userId)
            ->withdrawals()
            ->completed();

        if ($from && $to) {
            $query->forPeriod($from, $to);
        }

        return abs($query->sum('amount'));
    }

    /**
     * دریافت آخرین تراکنش کاربر
     */
    public static function lastTransaction($userId)
    {
        return static::where('user_id', $userId)
            ->latestFirst()
            ->first();
    }
}