<?php

namespace App\Services;

use App\Models\ServerIp;
use App\Models\Server;
use App\Models\User;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class IpManagementService
{
    protected $walletService;

    public function __construct(WalletService $walletService)
    {
        $this->walletService = $walletService;
    }

    /**
     * ایجاد Floating IP جدید
     */
    public function createFloatingIp(Server $server, string $type = 'ipv4')
    {
        // بررسی وضعیت سرور (باید خاموش باشه)
        if (!in_array($server->status, ['stopped', 'off', 'suspended'])) {
            throw new \Exception('برای افزودن Floating IP، سرور باید خاموش باشد');
        }

        // بررسی provider
        if ($server->provider === 'vultr' && $type === 'ipv4') {
            throw new \Exception('Vultr اجازه افزودن IPv4 اضافه نمی‌دهد');
        }

        // بررسی موجودی
        $user = $server->user;
        $monthlyCost = $this->getFloatingIpMonthlyCost($server->provider, $type);
        
        if ($monthlyCost > 0 && !$this->walletService->hasEnoughBalance($user, $monthlyCost)) {
            throw new \Exception('موجودی کیف پول کافی نیست. حداقل ' . number_format($monthlyCost) . ' تومان نیاز است (هزینه ماهانه)');
        }

        // ساخت رکورد IP
        $ip = ServerIp::create([
            'server_id' => $server->id,
            'user_id' => $server->user_id,
            'ip' => '0.0.0.0',
            'type' => $type,
            'is_primary' => 0,
            'is_floating' => 1,
            'assigned_at' => now(),
            'status' => 'assigned',
            'provider' => $server->provider,
            'hourly_cost' => 0, // Floating IPs ماهانه هستند نه ساعتی
        ]);

        try {
            if ($server->provider === 'hetzner') {
                $createdIp = $this->createHetznerFloatingIp($server, $type);
                
                $ip->update([
                    'ip' => $createdIp['ip'],
                    'provider_ip_id' => $createdIp['id'],
                    'location' => $createdIp['location'] ?? null,
                    'status' => 'active',
                ]);

                // کسر هزینه ماهانه بلافاصله
                $this->walletService->deduct(
                    $user,
                    $monthlyCost,
                    'floating_ip_monthly',
                    "هزینه ماهانه Floating IP {$createdIp['ip']} ({$type})",
                    'ip',
                    $ip->id
                );

                Log::info('✅ Hetzner Floating IP Created', [
                    'server_id' => $server->id,
                    'ip_id' => $ip->id,
                    'ip' => $createdIp['ip'],
                    'type' => $type,
                    'monthly_cost' => $monthlyCost,
                ]);

            } elseif ($server->provider === 'vultr' && $type === 'ipv6') {
                $createdIp = $this->enableVultrIpv6($server);
                
                $ip->update([
                    'ip' => $createdIp['ip'],
                    'status' => 'active',
                ]);

                Log::info('✅ Vultr IPv6 Enabled', [
                    'server_id' => $server->id,
                    'ip_id' => $ip->id,
                    'ip' => $createdIp['ip'],
                ]);
            }

            return $ip;

        } catch (\Exception $e) {
            $ip->delete();
            
            Log::error('❌ Failed to create Floating IP', [
                'error' => $e->getMessage(),
                'server_id' => $server->id,
                'type' => $type,
            ]);
            
            throw $e;
        }
    }

    /**
     * حذف Floating IP (بدون جریمه)
     */
    public function deleteFloatingIp(ServerIp $ip)
    {
        if (!$ip->isFloating()) {
            throw new \Exception('این IP از نوع Floating نیست');
        }

        $server = $ip->server;
        
        // بررسی وضعیت سرور
        if (!in_array($server->status, ['stopped', 'off', 'suspended'])) {
            throw new \Exception('سرور باید خاموش باشد');
        }

        // حذف از Provider
        try {
            $this->deleteIpFromProvider($ip);
        } catch (\Exception $e) {
            Log::error('Failed to delete IP from provider', [
                'ip_id' => $ip->id,
                'error' => $e->getMessage(),
            ]);
        }

        // حذف از دیتابیس
        $ip->detached_at = now();
        $ip->status = 'deleted';
        $ip->save();
        $ip->delete();

        Log::info('🗑️ Floating IP Deleted (No Penalty)', [
            'ip_id' => $ip->id,
            'ip' => $ip->ip,
            'server_id' => $server->id,
        ]);

        return true;
    }

    /**
     * حذف دائمی Detached Primary IP
     */
    public function deletePermanentlyDetachedIp(ServerIp $ip)
    {
        if (!$ip->isPrimary() || $ip->status !== 'detached') {
            throw new \Exception('فقط Primary IP های Detached قابل حذف دائمی هستند');
        }

        // حذف از Provider
        try {
            $this->deleteIpFromProvider($ip);
        } catch (\Exception $e) {
            Log::error('Failed to delete detached IP from provider', [
                'ip_id' => $ip->id,
                'error' => $e->getMessage(),
            ]);
        }

        // محاسبه هزینه تا الان
        $costSinceDetach = $ip->calculateDetachedCost();

        // حذف از دیتابیس
        $ip->status = 'deleted';
        $ip->save();
        $ip->delete();

        Log::info('🗑️ Detached Primary IP Permanently Deleted', [
            'ip_id' => $ip->id,
            'ip' => $ip->ip,
            'user_id' => $ip->user_id,
            'cost_since_detach' => $costSinceDetach,
            'hours_detached' => $ip->getHoursSinceDetach(),
        ]);

        return true;
    }

    /**
     * اتصال مجدد Detached IP به سرور
     */
/**
 * Reassign یک Detached IP به سرور دیگه
 */
public function reassignDetachedIp(Server $server, ServerIp $ip)
{
    if ($ip->status !== 'detached') {
        throw new \Exception('این IP در حالت Detached نیست');
    }

    if ($ip->user_id !== $server->user_id) {
        throw new \Exception('IP و سرور متعلق به کاربرهای مختلف هستند');
    }

    // چک Location
    if ($server->region !== $ip->location) {
        throw new \Exception("IP در Location {$ip->location} است ولی سرور در {$server->region}");
    }

    // چک وضعیت سرور
    if (!in_array($server->status, ['stopped', 'off'])) {
        throw new \Exception('سرور باید خاموش باشد');
    }

    // چک duplicate
    $existingIp = ServerIp::where('server_id', $server->id)
        ->where('type', $ip->type)
        ->where('is_primary', 1)
        ->whereIn('status', ['active', 'assigned'])
        ->first();

    if ($existingIp) {
        throw new \Exception("این سرور قبلاً یک Primary {$ip->type} دارد");
    }

    // ✅ محاسبه و کسر هزینه Detached (قبل از Assign)
    $detachedCost = 0;
    if ($ip->detached_at && $ip->type === 'ipv4') {
        $detachedCost = $ip->calculateDetachedCost();
        
        if ($detachedCost > 0) {
            $user = User::find($ip->user_id);
            if ($user) {
                // کسر از کیف پول
                $user->decrement('wallet_balance', $detachedCost);
                
                Log::info('💰 Detached IP cost charged', [
                    'user_id' => $user->id,
                    'ip_id' => $ip->id,
                    'ip' => $ip->ip,
                    'cost' => $detachedCost,
                    'hours' => $ip->getHoursSinceDetach(),
                ]);
            }
        }
    }

    // ✅ Assign کردن در Hetzner API
    try {
        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . config('services.hetzner.api_token'),
            'Content-Type' => 'application/json',
        ])->post("https://api.hetzner.cloud/v1/primary_ips/{$ip->provider_ip_id}/actions/assign", [
            'assignee_id' => (int) $server->provider_id,
            'assignee_type' => 'server',
        ]);

        if (!$response->successful()) {
            $error = $response->json();
            Log::error('❌ Hetzner Assign API Error', [
                'ip_id' => $ip->id,
                'server_id' => $server->id,
                'response' => $error,
                'status' => $response->status(),
            ]);
            
            throw new \Exception($error['error']['message'] ?? 'خطا در Assign از Hetzner');
        }

        Log::info('✅ Hetzner Assign API Success', [
            'ip_id' => $ip->id,
            'ip' => $ip->ip,
            'server_id' => $server->id,
            'server_name' => $server->name,
        ]);

    } catch (\Exception $e) {
        throw new \Exception('خطا در تخصیص IP به سرور: ' . $e->getMessage());
    }

    // ✅ آپدیت IP در دیتابیس
    $ip->update([
        'server_id' => $server->id,
        'status' => 'active',
        'assigned_at' => now(),
        'detached_at' => null, // ✅ ریست detached_at
    ]);

    // ✅ آپدیت IP اصلی سرور (اگه IPv4 بود)
    if ($ip->type === 'ipv4') {
        $server->update([
            'ip_address' => $ip->ip,
        ]);
    }

    Log::info('✅ IP reassigned to server', [
        'ip_id' => $ip->id,
        'ip' => $ip->ip,
        'server_id' => $server->id,
        'server_name' => $server->name,
        'detached_cost' => $detachedCost,
    ]);

    return $ip;
}


/**
 * ✅ ساخت Primary IP جدید توسط کاربر (از Panel)
 * IP به صورت Detached ساخته میشه و بعداً با دکمه Assign وصل میشه
 */
public function createPrimaryIp(string $type, string $location, ?string $name = null, ?int $serverId = null)
{
    // Validation
    if (!in_array($type, ['ipv4', 'ipv6'])) {
        throw new \Exception('نوع IP نامعتبر است');
    }

    if (empty($location)) {
        throw new \Exception('Location الزامی است');
    }

    // هزینه
    $hourlyCost = ($type === 'ipv4') ? ServerIp::HETZNER_PRIMARY_IPV4_HOURLY : 0;

    // ✅ ساخت Primary IP در Hetzner (بدون Assign)
    try {
        $payload = [
            'type' => $type,
            'name' => $name ?? 'primary-ip-' . uniqid(),
            'assignee_type' => 'server',
            'auto_delete' => false,
            'location' => $location, // ✅ فقط location
        ];

        Log::info('🔵 Creating Primary IP in Hetzner (Detached)', $payload);

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . config('services.hetzner.api_token'),
            'Content-Type' => 'application/json',
        ])->post('https://api.hetzner.cloud/v1/primary_ips', $payload);

        if (!$response->successful()) {
            $error = $response->json();
            Log::error('❌ Hetzner Create Primary IP Error', [
                'payload' => $payload,
                'response' => $error,
                'status' => $response->status(),
            ]);

            throw new \Exception($error['error']['message'] ?? 'خطا در ساخت Primary IP از Hetzner');
        }

        $result = $response->json();
        $primaryIp = $result['primary_ip'] ?? null;

        if (!$primaryIp) {
            throw new \Exception('پاسخ نامعتبر از Hetzner API');
        }

        // ✅ دریافت Location از response
        $actualLocation = $primaryIp['datacenter']['location']['name'] ?? $location;

        Log::info('✅ Primary IP created in Hetzner', [
            'hetzner_id' => $primaryIp['id'],
            'ip' => $primaryIp['ip'],
            'type' => $primaryIp['type'],
            'location' => $actualLocation,
        ]);

    } catch (\Exception $e) {
        throw new \Exception('خطا در ساخت Primary IP: ' . $e->getMessage());
    }

    // ✅ ذخیره در دیتابیس (همیشه Detached)
    $ipModel = ServerIp::create([
        'user_id' => auth()->id(),
        'server_id' => null, // ✅ همیشه null
        'ip' => $primaryIp['ip'],
        'type' => $primaryIp['type'],
        'is_primary' => 1,
        'is_floating' => 0,
        'status' => 'detached', // ✅ همیشه detached
        'assigned_at' => null,
        'detached_at' => now(), // ✅ از همین الان شروع به محاسبه هزینه
        'provider' => 'hetzner',
        'provider_ip_id' => $primaryIp['id'],
        'location' => $actualLocation,
        'datacenter' => $primaryIp['datacenter']['name'] ?? null,
        'hourly_cost' => $hourlyCost,
    ]);

    Log::info('✅ Primary IP saved to database (Detached)', [
        'ip_id' => $ipModel->id,
        'ip' => $ipModel->ip,
        'type' => $ipModel->type,
        'hourly_cost' => $hourlyCost,
        'status' => 'detached',
    ]);

    return $ipModel;
}

    /**
     * ✅ ایجاد Primary IP جدید در Hetzner
     */
    private function createNewPrimaryIp(Server $server, string $type = 'ipv4')
    {
        $apiToken = config('services.hetzner.api_token');
        
        if (!$apiToken) {
            throw new \Exception('Hetzner API Token تنظیم نشده است');
        }

        // دریافت Location سرور
        $location = null;
        try {
            $serverResponse = Http::withHeaders([
                'Authorization' => 'Bearer ' . $apiToken,
            ])->get("https://api.hetzner.cloud/v1/servers/{$server->provider_id}");

            if ($serverResponse->successful()) {
                $serverData = $serverResponse->json('server');
                $location = $serverData['datacenter']['location']['name'] ?? null;
            }
        } catch (\Exception $e) {
            Log::warning('Failed to get server location', ['error' => $e->getMessage()]);
        }

        Log::info('Creating Primary IP', [
            'server_id' => $server->id,
            'provider_id' => $server->provider_id,
            'type' => $type,
            'location' => $location,
        ]);

        // ساخت Primary IP بدون assignee
        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $apiToken,
            'Content-Type' => 'application/json',
        ])->post('https://api.hetzner.cloud/v1/primary_ips', [
            'type' => $type,
            'name' => "primary-{$type}-{$server->name}-" . time(),
            'auto_delete' => false,
        ]);

        if (!$response->successful()) {
            $error = $response->json();
            Log::error('Failed to create Primary IP', ['server_id' => $server->id, 'error' => $error]);
            throw new \Exception($error['error']['message'] ?? 'خطا در ایجاد Primary IP');
        }

        $data = $response->json();
        $primaryIp = $data['primary_ip'];

        // Assign به سرور
        $assignResponse = Http::withHeaders([
            'Authorization' => 'Bearer ' . $apiToken,
            'Content-Type' => 'application/json',
        ])->post("https://api.hetzner.cloud/v1/primary_ips/{$primaryIp['id']}/actions/assign", [
            'assignee_id' => (int)$server->provider_id,
        ]);

        if (!$assignResponse->successful()) {
            // حذف IP اگر assign نشد
            try {
                Http::withHeaders(['Authorization' => 'Bearer ' . $apiToken])
                    ->delete("https://api.hetzner.cloud/v1/primary_ips/{$primaryIp['id']}");
            } catch (\Exception $e) {
                Log::error('Failed to cleanup Primary IP', ['error' => $e->getMessage()]);
            }
            
            $assignError = $assignResponse->json();
            Log::error('Failed to assign Primary IP', [
                'server_id' => $server->id,
                'primary_ip_id' => $primaryIp['id'],
                'error' => $assignError,
            ]);
            
            throw new \Exception('خطا در تخصیص IP به سرور: ' . ($assignError['error']['message'] ?? 'خطای نامشخص'));
        }

        // تعیین هزینه ساعتی
        $hourlyCost = ($type === 'ipv4') 
           ? ServerIp::HETZNER_PRIMARY_IPV4_HOURLY  // 250 تومان
           : 0; // ✅ IPv6 رایگان
 
        // ذخیره در دیتابیس
        $newIp = ServerIp::create([
            'server_id' => $server->id,
            'user_id' => $server->user_id,
            'ip' => $primaryIp['ip'],
            'type' => $type,
            'is_primary' => 1,
            'is_floating' => 0,
            'assigned_at' => now(),
            'status' => 'active',
            'provider' => 'hetzner',
            'provider_ip_id' => $primaryIp['id'],
            'location' => $location,
            'hourly_cost' => $hourlyCost,
        ]);

        Log::info('✅ New Hetzner Primary IP Created', [
            'server_id' => $server->id,
            'ip_id' => $newIp->id,
            'ip' => $newIp->ip,
            'provider_ip_id' => $primaryIp['id'],
            'location' => $location,
            'hourly_cost' => $hourlyCost,
        ]);

        return $newIp;
    }

    /**
     * ایجاد Floating IP در Hetzner
     */
    private function createHetznerFloatingIp(Server $server, string $type)
    {
        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . config('services.hetzner.api_token'),
            'Content-Type' => 'application/json',
        ])->post('https://api.hetzner.cloud/v1/floating_ips', [
            'type' => $type,
            'server' => (int)$server->provider_id,
            'description' => "Floating IP for server {$server->name}",
        ]);

        if (!$response->successful()) {
            $error = $response->json();
            throw new \Exception($error['error']['message'] ?? 'خطا در ایجاد Floating IP');
        }

        $data = $response->json();
        $floatingIp = $data['floating_ip'];
        
        return [
            'id' => $floatingIp['id'],
            'ip' => $floatingIp['ip'],
            'location' => $floatingIp['location']['name'] ?? null,
        ];
    }

    /**
     * فعال‌سازی IPv6 در Vultr
     */
    private function enableVultrIpv6(Server $server)
    {
        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . config('services.vultr.api_key'),
        ])->get("https://api.vultr.com/v2/instances/{$server->provider_id}/ipv6");

        if (!$response->successful()) {
            throw new \Exception('خطا در دریافت IPv6');
        }

        $ipv6s = $response->json('ipv6') ?? [];
        
        if (empty($ipv6s)) {
            throw new \Exception('این سرور IPv6 ندارد');
        }

        return [
            'ip' => $ipv6s[0]['ip'] ?? 'auto',
            'id' => null,
        ];
    }

    /**
     * Unassign Primary IP از Hetzner
     */
    private function unassignHetznerPrimaryIp(ServerIp $ip)
    {
        if (!$ip->provider_ip_id) {
            return;
        }

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . config('services.hetzner.api_token'),
        ])->post("https://api.hetzner.cloud/v1/primary_ips/{$ip->provider_ip_id}/actions/unassign");

        if (!$response->successful()) {
            throw new \Exception('خطا در جداسازی IP از Hetzner');
        }

        Log::info('Primary IP unassigned from Hetzner', [
            'ip_id' => $ip->id,
            'provider_ip_id' => $ip->provider_ip_id,
        ]);
    }

    /**
     * Assign Primary IP به سرور Hetzner
     */
    private function assignHetznerPrimaryIp(Server $server, ServerIp $ip)
    {
        if (!$ip->provider_ip_id) {
            throw new \Exception('IP فاقد provider_ip_id است');
        }

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . config('services.hetzner.api_token'),
            'Content-Type' => 'application/json',
        ])->post("https://api.hetzner.cloud/v1/primary_ips/{$ip->provider_ip_id}/actions/assign", [
            'assignee_id' => (int)$server->provider_id,
        ]);

        if (!$response->successful()) {
            $error = $response->json();
            
            Log::error('Failed to assign Primary IP', [
                'ip_id' => $ip->id,
                'provider_ip_id' => $ip->provider_ip_id,
                'server_provider_id' => $server->provider_id,
                'error' => $error,
            ]);
            
            throw new \Exception('خطا در تخصیص IP به سرور: ' . ($error['error']['message'] ?? 'خطای نامشخص'));
        }

        Log::info('Primary IP assigned to server', [
            'server_id' => $server->id,
            'ip_id' => $ip->id,
            'provider_ip_id' => $ip->provider_ip_id,
        ]);
    }


public function unassignPrimaryIp(ServerIp $ip)
{
    if (!$ip->isPrimary()) {
        throw new \Exception('این IP از نوع Primary نیست');
    }

    if ($ip->status === 'detached') {
        throw new \Exception('این IP قبلاً جدا شده است');
    }

    $server = $ip->server;

    if (!$server) {
        throw new \Exception('IP به سروری متصل نیست');
    }

    // چک کردن وضعیت سرور
    if (!in_array($server->status, ['stopped', 'off'])) {
        throw new \Exception('سرور باید خاموش باشد. لطفاً ابتدا سرور را خاموش کنید.');
    }

    // Unassign از Hetzner
    try {
        $this->unassignHetznerPrimaryIp($ip);
    } catch (\Exception $e) {
        Log::error('Failed to unassign from Hetzner', [
            'ip_id' => $ip->id,
            'error' => $e->getMessage(),
        ]);
        throw new \Exception('خطا در جداسازی IP از Hetzner: ' . $e->getMessage());
    }

    // آپدیت رکورد
    $ip->update([
        'server_id' => null,
        'is_primary' => 1,
        'detached_at' => now(),
        'assigned_at' => null,
        'status' => 'detached',
    ]);

    Log::info('✅ Primary IP unassigned from server', [
        'ip_id' => $ip->id,
        'ip' => $ip->ip,
        'server_id' => $server->id,
    ]);

    return $ip;
}



    /**
     * حذف IP از Provider
     */
    private function deleteIpFromProvider(ServerIp $ip)
    {
        if ($ip->provider === 'hetzner' && $ip->provider_ip_id) {
            $endpoint = $ip->is_floating 
                ? "floating_ips/{$ip->provider_ip_id}"
                : "primary_ips/{$ip->provider_ip_id}";
                
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . config('services.hetzner.api_token'),
            ])->delete("https://api.hetzner.cloud/v1/{$endpoint}");

            if (!$response->successful()) {
                throw new \Exception('خطا در حذف IP از Hetzner');
            }

            Log::info('IP deleted from Hetzner', [
                'ip_id' => $ip->id,
                'provider_ip_id' => $ip->provider_ip_id,
                'type' => $ip->is_floating ? 'floating' : 'primary',
            ]);
        }
    }

    /**
     * محاسبه هزینه ماهانه Floating IP
     */
    private function getFloatingIpMonthlyCost(string $provider, string $type): float
    {
        if ($provider === 'hetzner') {
            if ($type === 'ipv4') {
                return ServerIp::HETZNER_FLOATING_IPV4_MONTHLY;
            }
            if ($type === 'ipv6') {
                return ServerIp::HETZNER_FLOATING_IPV6_MONTHLY;
            }
        }

        if ($provider === 'vultr') {
            return ServerIp::VULTR_IPV4_FLOATING_HOURLY * 730;
        }

        return 0;
    }

    /**
     * بررسی امکان افزودن Floating IP
     */
    public function canAddFloatingIp(Server $server, string $type = 'ipv4'): array
    {
        $errors = [];

        if (!in_array($server->status, ['stopped', 'off', 'suspended'])) {
            $errors[] = 'سرور باید خاموش باشد';
        }

        if ($server->provider === 'vultr' && $type === 'ipv4') {
            $errors[] = 'Vultr اجازه افزودن IPv4 اضافه نمی‌دهد';
        }

        $user = $server->user;
        $cost = $this->getFloatingIpMonthlyCost($server->provider, $type);
        
        if ($cost > 0 && !$this->walletService->hasEnoughBalance($user, $cost)) {
            $errors[] = 'موجودی کیف پول کافی نیست (' . number_format($cost) . ' تومان)';
        }

        return [
            'can_add' => empty($errors),
            'errors' => $errors,
            'monthly_cost' => $cost,
        ];
    }


    
    /**
     * ✅ ساخت Primary IP مستقل - نسخه نهایی
     */
/**
 * ✅ ساخت Primary IP مستقل - با datacenter (بدون سرور)
 */
public function createStandalonePrimaryIp(User $user, string $type, string $location, ?string $name = null, ?int $serverId = null)
{
    $apiToken = config('services.hetzner.api_token');
    
    if (!$apiToken) {
        throw new \Exception('Hetzner API Token تنظیم نشده است');
    }

    Log::info('🔵 Creating Standalone Primary IP', [
        'user_id' => $user->id,
        'type' => $type,
        'location' => $location,
        'server_id' => $serverId,
    ]);

    // ✅ دریافت datacenter از location
    $datacenter = $this->getDatacenterFromLocation($location);
    
    if (!$datacenter) {
        throw new \Exception("Location '{$location}' معتبر نیست");
    }

    // ✅ ساخت Primary IP با datacenter (بدون assignee_id)
    $requestBody = [
        'type' => $type,
        'name' => $name ?? "primary-{$type}-" . time(),
        'datacenter' => $datacenter,
        'assignee_type' => 'server',
        'auto_delete' => false,
    ];

    $jsonBody = json_encode($requestBody);

    Log::info('📤 Creating Primary IP (detached)', [
        'datacenter' => $datacenter,
        'type' => $type,
    ]);

    // cURL
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => 'https://api.hetzner.cloud/v1/primary_ips',
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $jsonBody,
        CURLOPT_HTTPHEADER => [
            'Authorization: Bearer ' . $apiToken,
            'Content-Type: application/json',
        ],
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_TIMEOUT => 30,
    ]);

    $response = curl_exec($ch);
    $statusCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    Log::info('📥 Hetzner Response', [
        'status' => $statusCode,
        'response' => $response,
    ]);

    if ($curlError) {
        throw new \Exception('cURL Error: ' . $curlError);
    }

    $data = json_decode($response, true);

    if ($statusCode !== 201) {
        $error = $data['error'] ?? [];
        Log::error('❌ Hetzner Error', ['error' => $error]);
        throw new \Exception($error['message'] ?? 'خطا در ایجاد Primary IP');
    }

    $primaryIp = $data['primary_ip'];

    Log::info('✅ Primary IP created (detached)', [
        'ip' => $primaryIp['ip'],
        'id' => $primaryIp['id'],
        'datacenter' => $datacenter,
    ]);

    // هزینه ساعتی
    $hourlyCost = ($type === 'ipv4') 
      ? ServerIp::HETZNER_PRIMARY_IPV4_HOURLY  // 250 تومان
      : 0; // ✅ IPv6 رایگان

    // ذخیره به صورت detached
    $newIp = ServerIp::create([
        'user_id' => $user->id,
        'server_id' => null,
        'ip' => $primaryIp['ip'],
        'type' => $type,
        'is_primary' => 1,
        'is_floating' => 0,
        'assigned_at' => null,
        'status' => 'detached',
        'detached_at' => now(),
        'provider' => 'hetzner',
        'provider_ip_id' => (string)$primaryIp['id'],
        'location' => $primaryIp['datacenter']['location']['name'] ?? null,
        'datacenter' => $primaryIp['datacenter']['name'] ?? null,
        'hourly_cost' => $hourlyCost,
        'rdns' => $primaryIp['dns_ptr'][0]['dns_ptr'] ?? null,
    ]);

    Log::info('✅ Primary IP saved as detached', ['ip_id' => $newIp->id]);


    // ✅ کسر هزینه ساعت اول
  $this->walletService->deduct(
      $user,
      $hourlyCost,
      'primary_ip_hourly',
       "هزینه ساعت اول Primary IP {$newIp->ip} ({$type}) - Location: {$location}",
      'ip',
      $newIp->id
    );

   Log::info('💰 First hour cost deducted', [
      'ip_id' => $newIp->id,
      'cost' => $hourlyCost,
   ]);

    // ✅ اگه کاربر سرور انتخاب کرده، الان assign کن
    if ($serverId) {
        try {
            $server = Server::findOrFail($serverId);
            
            if ($server->user_id !== $user->id) {
                throw new \Exception('این سرور متعلق به شما نیست');
            }

            if (!in_array($server->status, ['stopped', 'off'])) {
                throw new \Exception('سرور باید خاموش باشد');
            }

            // چک location
            $serverLocation = $this->getServerLocation($server);
            $ipLocation = $primaryIp['datacenter']['location']['name'] ?? null;
            
            if ($serverLocation && $ipLocation && $serverLocation !== $ipLocation) {
                throw new \Exception("این IP فقط در Location {$ipLocation} قابل استفاده است، ولی سرور در {$serverLocation} است");
            }

            // چک duplicate
            $existingIp = ServerIp::where('server_id', $server->id)
                ->where('type', $type)
                ->where('is_primary', 1)
                ->whereIn('status', ['active', 'assigned'])
                ->first();
            
            if ($existingIp) {
                throw new \Exception("این سرور قبلاً یک Primary {$type} دارد");
            }

            // Assign
            $this->assignHetznerPrimaryIp($server, $newIp);
            
            $newIp->update([
                'server_id' => $server->id,
                'status' => 'active',
                'assigned_at' => now(),
                'detached_at' => null,
            ]);

            Log::info('✅ Primary IP assigned to server', [
                'server_id' => $server->id,
            ]);
        } catch (\Exception $e) {
            Log::warning('⚠️ Failed to auto-assign', [
                'error' => $e->getMessage(),
            ]);
            // IP ساخته شده، فقط assign نشد
        }
    }

    return $newIp;
}

    /**
     * ✅ دریافت Location سرور
     */
    private function getServerLocation(Server $server): ?string
    {
        $apiToken = config('services.hetzner.api_token');
        
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $apiToken,
            ])->get("https://api.hetzner.cloud/v1/servers/{$server->provider_id}");

            if ($response->successful()) {
                $serverData = $response->json('server');
                return $serverData['datacenter']['location']['name'] ?? null;
            }
        } catch (\Exception $e) {
            Log::warning('Failed to get server location', ['error' => $e->getMessage()]);
        }
        
        return null;
    }

/**
 * ✅ دریافت Datacenter از Location
 */
private function getDatacenterFromLocation(string $location): ?string
{
    $cleanLocation = $this->cleanLocationName($location);
    
    $locationMap = [
        // نام کامل
        'nuremberg' => 'nbg1-dc3',
        'falkenstein' => 'fsn1-dc14',
        'helsinki' => 'hel1-dc2',
        'ashburn' => 'ash-dc1',
        'hillsboro' => 'hil-dc1',
        'singapore' => 'sin-dc1',
        
        // ✅ کدهای کوتاه Hetzner
        'nbg1' => 'nbg1-dc3',
        'fsn1' => 'fsn1-dc14',
        'hel1' => 'hel1-dc2',
        'ash' => 'ash-dc1',
        'hil' => 'hil-dc1',
        'sin' => 'sin-dc1',
    ];

    $result = $locationMap[$cleanLocation] ?? null;
    
    Log::info('🔍 Location to Datacenter mapping', [
        'input' => $location,
        'cleaned' => $cleanLocation,
        'datacenter' => $result,
    ]);

    return $result;
}

/**
 * ✅ تمیز کردن نام Location
 */
private function cleanLocationName(string $location): string
{
    $location = preg_replace('/\s*\(.*?\)\s*/', '', $location);
    return strtolower(trim($location));
}
}