<?php

namespace App\Services;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class LimitService
{
    /**
     * دریافت محدودیت‌های کاربر
     */
    public function getLimits()
    {
        return [
            'projects' => 5,
            'servers' => 5,
            'ips' => 10,
            'deletions' => 15
        ];
    }

    /**
     * دریافت استفاده کاربر فعلی
     */
    public function getUsage($userId = null)
    {
        $userId = $userId ?? Auth::id();

        return [
            'projects' => $this->getUserProjectsCount($userId),
            'servers' => $this->getUserServersCount($userId),
            'ips' => $this->getUserIpsCount($userId),
            'deletions' => $this->getUserServerDeletionsCount($userId)
        ];
    }

    /**
     * بررسی آیا می‌توان پروژه جدید ایجاد کرد
     */
    public function canCreateProject($userId = null): bool
    {
        $userId = $userId ?? Auth::id();
        $limits = $this->getLimits();
        $usage = $this->getUsage($userId);

        return $usage['projects'] < $limits['projects'];
    }

    /**
     * بررسی آیا می‌توان سرور جدید ایجاد کرد
     * ✅ برای تمام providers (Hetzner, Arvan, Vultr, Parspack, ...)
     */
    public function canCreateServer($userId = null): bool
    {
        $userId = $userId ?? Auth::id();
        $limits = $this->getLimits();
        $usage = $this->getUsage($userId);

        return $usage['servers'] < $limits['servers'];
    }

    /**
     * بررسی آیا می‌توان IP جدید ایجاد کرد
     */
    public function canCreateIP($userId = null): bool
    {
        $userId = $userId ?? Auth::id();
        $limits = $this->getLimits();
        $usage = $this->getUsage($userId);

        return $usage['ips'] < $limits['ips'];
    }

    /**
     * بررسی آیا می‌توان سرور حذف کرد (ماهانه محدود)
     * ✅ هر ماه رفرش می‌شه
     */
    public function canDelete($userId = null): bool
    {
        $userId = $userId ?? Auth::id();
        $limits = $this->getLimits();
        $usage = $this->getUsage($userId);

        return $usage['deletions'] < $limits['deletions'];
    }

    /**
     * ثبت حذف سرور
     */
    public function logDeletion($userId, $serverId)
    {
        if (!DB::getSchemaBuilder()->hasTable('deletion_logs')) {
            return false;
        }

        DB::table('deletion_logs')->insert([
            'user_id' => $userId,
            'server_id' => $serverId,
            'type' => 'server',
            'created_at' => now(),
            'updated_at' => now()
        ]);

        return true;
    }

    /**
     * دریافت اطلاعات کامل محدودیت برای نمایش
     * ✅ این برای صفحه محدودیت‌ها استفاده کن
     */
    public function getLimitInfo($userId = null): array
    {
        $userId = $userId ?? Auth::id();
        $limits = $this->getLimits();
        $usage = $this->getUsage($userId);

        return [
            'projects' => [
                'used' => $usage['projects'],
                'limit' => $limits['projects'],
                'remaining' => $limits['projects'] - $usage['projects'],
                'percentage' => round(($usage['projects'] / $limits['projects']) * 100, 2),
                'can_create' => $this->canCreateProject($userId),
            ],
            'servers' => [
                'used' => $usage['servers'],
                'limit' => $limits['servers'],
                'remaining' => $limits['servers'] - $usage['servers'],
                'percentage' => round(($usage['servers'] / $limits['servers']) * 100, 2),
                'can_create' => $this->canCreateServer($userId),
            ],
            'ips' => [
                'used' => $usage['ips'],
                'limit' => $limits['ips'],
                'remaining' => $limits['ips'] - $usage['ips'],
                'percentage' => round(($usage['ips'] / $limits['ips']) * 100, 2),
                'can_create' => $this->canCreateIP($userId),
            ],
            'deletions' => [
                'used' => $usage['deletions'],
                'limit' => $limits['deletions'],
                'remaining' => $limits['deletions'] - $usage['deletions'],
                'percentage' => round(($usage['deletions'] / $limits['deletions']) * 100, 2),
                'can_delete' => $this->canDelete($userId),
            ],
        ];
    }

    /**
     * شمارش تعداد پروژه‌های کاربر
     */
    private function getUserProjectsCount($userId)
    {
        return DB::table('projects')
            ->where('user_id', $userId)
            ->count();
    }

    /**
     * شمارش تعداد سرورهای کاربر (از تمام providers)
     * ✅ شمارش: Hetzner + Arvan + Vultr + Parspack + ...
     */
    private function getUserServersCount($userId)
    {
        return DB::table('servers')
            ->where('user_id', $userId)
            ->count();
    }

    /**
     * شمارش تعداد IP های کاربر
     */
    private function getUserIpsCount($userId)
    {
        return DB::table('server_ips')
            ->where('user_id', $userId)
            ->count();
    }

    /**
     * شمارش تعداد حذف سرورهای ماه جاری
     * ✅ فقط حذف‌های این ماه
     * ✅ خودکار ماه بعد رفرش می‌شه
     */
    private function getUserServerDeletionsCount($userId)
    {
        if (!DB::getSchemaBuilder()->hasTable('deletion_logs')) {
            return 0;
        }

        return DB::table('deletion_logs')
            ->where('user_id', $userId)
            ->where('type', 'server')
            ->whereYear('created_at', now()->year)
            ->whereMonth('created_at', now()->month)
            ->count();
    }
}