<?php

namespace App\Services;

use App\Models\User;
use App\Models\WalletTransaction;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class WalletService
{
    /**
     * واریز به کیف پول
     */
public function deposit(
    User $user,
    float $amount,
    string $description,
    $invoiceId = null,
    array $metadata = []
) {
    return DB::transaction(function () use ($user, $amount, $description, $invoiceId, $metadata) {
        $balanceBefore = $user->wallet_balance;
        $balanceAfter = $balanceBefore + $amount;

        $user->wallet_balance = $balanceAfter;
        $user->save();

        // ✅ اضافه کردن ref_type و ref_id برای invoice
        $transaction = WalletTransaction::create([
            'user_id' => $user->id,
            'amount' => $amount,
            'type' => 'deposit',
            'status' => 'completed',
            'description' => $description,
            'invoice_id' => $invoiceId,
            'ref_type' => $invoiceId ? 'invoice' : null,  // ✅ اضافه شد
            'ref_id' => $invoiceId,                         // ✅ اضافه شد
            'balance_before' => $balanceBefore,
            'balance_after' => $balanceAfter,
            'metadata' => $metadata,
        ]);

        Log::info('💰 Wallet Deposit', [
            'user_id' => $user->id,
            'amount' => $amount,
            'invoice_id' => $invoiceId,
            'transaction_id' => $transaction->id,
        ]);

        return $transaction;
    });
}

    /**
     * کسر از کیف پول
     */
    public function deduct(
        User $user,
        float $amount,
        string $type,
        string $description,
        $refType = null,
        $refId = null,
        array $metadata = []
    ) {
        if ($user->wallet_balance < $amount) {
            Log::warning('❌ Insufficient Balance', [
                'user_id' => $user->id,
                'required' => $amount,
                'available' => $user->wallet_balance,
            ]);
            return null;
        }

        return DB::transaction(function () use ($user, $amount, $type, $description, $refType, $refId, $metadata) {
            $balanceBefore = $user->wallet_balance;
            $balanceAfter = $balanceBefore - $amount;

            // بروزرسانی موجودی
            $user->wallet_balance = $balanceAfter;
            $user->save();

            // ثبت تراکنش
            $transaction = WalletTransaction::create([
                'user_id' => $user->id,
                'amount' => -$amount,
                'type' => $type,
                'status' => 'completed',
                'description' => $description,
                'ref_type' => $refType,
                'ref_id' => $refId,
                'balance_before' => $balanceBefore,
                'balance_after' => $balanceAfter,
                'metadata' => $metadata,
            ]);

            Log::info('💸 Wallet Deducted', [
                'user_id' => $user->id,
                'amount' => $amount,
                'type' => $type,
                'balance_after' => $balanceAfter,
                'transaction_id' => $transaction->id,
            ]);

            return $transaction;
        });
    }

    /**
     * کسر هزینه ساعتی سرور
     */
    public function deductServerHourlyCost(
        User $user,
        int $serverId,
        string $serverName,
        float $cost,
        string $provider
    ) {
        return $this->deduct(
            $user,
            $cost,
            'server_hourly',
            "هزینه ساعتی سرور {$serverName}",
            'server',
            $serverId,
            [
                'server_id' => $serverId,
                'server_name' => $serverName,
                'provider' => $provider,
                'cost_type' => 'hourly',
            ]
        );
    }

    /**
     * کسر هزینه ساعتی IP
     */
    public function deductIpHourlyCost(
        User $user,
        int $ipId,
        string $ipAddress,
        int $serverId,
        float $cost,
        string $ipType
    ) {
        return $this->deduct(
            $user,
            $cost,
            'ip_hourly',
            "هزینه ساعتی IP {$ipAddress} (سرور #{$serverId})",
            'ip',
            $ipId,
            [
                'ip_id' => $ipId,
                'ip_address' => $ipAddress,
                'server_id' => $serverId,
                'ip_type' => $ipType,
                'cost_type' => 'hourly',
            ]
        );
    }

    /**
     * اعمال جریمه
     */
    public function applyPenalty(
        User $user,
        float $amount,
        string $description,
        $refType = null,
        $refId = null
    ) {
        return $this->deduct(
            $user,
            $amount,
            'penalty',
            $description,
            $refType,
            $refId,
            ['penalty_type' => 'deletion', 'applied_at' => now()]
        );
    }

    /**
     * دریافت تراکنش‌های کاربر
     */
    public function getUserTransactions(
        User $user,
        $type = null,
        $from = null,
        $to = null,
        int $perPage = 20
    ) {
        $from = $from ? $from : now()->startOfMonth()->toDateString();
        $to = $to ? $to : now()->endOfMonth()->toDateString();

        $query = $user->walletTransactions()->orderBy('created_at', 'desc');

        if ($type) {
            $query->where('type', $type);
        }

        $query->whereBetween('created_at', [
            $from . ' 00:00:00',
            $to . ' 23:59:59'
        ]);

        return $query->paginate($perPage);
    }

    /**
     * دریافت آمار کاربر
     */
    public function getUserStats(User $user, $from = null, $to = null): array
    {
        $from = $from ? $from : now()->startOfMonth()->toDateString();
        $to = $to ? $to : now()->endOfMonth()->toDateString();

        $fromDateTime = $from . ' 00:00:00';
        $toDateTime = $to . ' 23:59:59';

        $currentBalance = $user->wallet_balance ? $user->wallet_balance : 0;
        $totalDeposits = $this->getTotalDeposits($user, $fromDateTime, $toDateTime);
        $totalCosts = $this->getTotalCost($user, $fromDateTime, $toDateTime);
        $serverCosts = $this->getCostByType($user, 'server_hourly', $fromDateTime, $toDateTime);
        $ipCosts = $this->getCostByType($user, 'ip_hourly', $fromDateTime, $toDateTime);
        
        $transactionCount = $user->walletTransactions()
            ->whereBetween('created_at', [$fromDateTime, $toDateTime])
            ->count();

        return [
            'current_balance' => $currentBalance,
            'total_deposits' => $totalDeposits,
            'total_costs' => $totalCosts,
            'server_costs' => $serverCosts,
            'ip_costs' => $ipCosts,
            'transaction_count' => $transactionCount,
            'period' => [
                'from' => $from,
                'to' => $to,
            ],
        ];
    }
    
    
    /**
 * دریافت تراکنش‌های مربوط به یک سرور خاص
 */
public function getServerTransactions(
    int $serverId,
    $from = null,
    $to = null,
    int $perPage = 20
) {
    $from = $from ? $from : now()->startOfMonth()->toDateString();
    $to = $to ? $to : now()->endOfMonth()->toDateString();

    return WalletTransaction::where(function($query) use ($serverId) {
            // تراکنش‌های مستقیم سرور
            $query->where('ref_type', 'server')
                  ->where('ref_id', $serverId);
        })
        ->orWhere(function($query) use ($serverId) {
            // تراکنش‌های IP های این سرور
            $query->where('ref_type', 'ip')
                  ->whereHas('serverIp', function($q) use ($serverId) {
                      $q->where('server_id', $serverId);
                  });
        })
        ->whereBetween('created_at', [
            $from . ' 00:00:00',
            $to . ' 23:59:59'
        ])
        ->where('status', 'completed')
        ->orderBy('created_at', 'desc')
        ->paginate($perPage);
}

/**
 * دریافت آمار هزینه‌های یک سرور خاص
 */
public function getServerCostStats(int $serverId, $from = null, $to = null): array
{
    $from = $from ? $from : now()->startOfMonth()->toDateString();
    $to = $to ? $to : now()->endOfMonth()->toDateString();

    $fromDateTime = $from . ' 00:00:00';
    $toDateTime = $to . ' 23:59:59';

    // هزینه‌های سرور
    $serverCosts = WalletTransaction::where('ref_type', 'server')
        ->where('ref_id', $serverId)
        ->where('status', 'completed')
        ->whereBetween('created_at', [$fromDateTime, $toDateTime])
        ->sum('amount');

    // هزینه‌های IP های این سرور
    $ipCosts = WalletTransaction::where('ref_type', 'ip')
        ->whereHas('serverIp', function($q) use ($serverId) {
            $q->where('server_id', $serverId);
        })
        ->where('status', 'completed')
        ->whereBetween('created_at', [$fromDateTime, $toDateTime])
        ->sum('amount');

    $totalCost = abs($serverCosts + $ipCosts);
    
    // تعداد تراکنش‌ها
    $transactionCount = WalletTransaction::where(function($query) use ($serverId) {
            $query->where('ref_type', 'server')
                  ->where('ref_id', $serverId);
        })
        ->orWhere(function($query) use ($serverId) {
            $query->where('ref_type', 'ip')
                  ->whereHas('serverIp', function($q) use ($serverId) {
                      $q->where('server_id', $serverId);
                  });
        })
        ->where('status', 'completed')
        ->whereBetween('created_at', [$fromDateTime, $toDateTime])
        ->count();

    return [
        'total_cost' => $totalCost,
        'server_costs' => abs($serverCosts),
        'ip_costs' => abs($ipCosts),
        'transaction_count' => $transactionCount,
        'period' => [
            'from' => $from,
            'to' => $to,
        ],
    ];
}

    

    /**
     * محاسبه مجموع واریزی‌ها
     */
    private function getTotalDeposits(User $user, $from, $to): float
    {
        return $user->walletTransactions()
            ->where('type', 'deposit')
            ->where('amount', '>', 0)
            ->where('status', 'completed')
            ->whereBetween('created_at', [$from, $to])
            ->sum('amount');
    }

    /**
     * محاسبه مجموع هزینه‌ها
     */
    private function getTotalCost(User $user, $from, $to): float
    {
        return abs(
            $user->walletTransactions()
                ->whereIn('type', [
                    'server_hourly',
                    'ip_hourly',
                    'server_create',
                    'ip_create',
                    'bandwidth_overage',
                    'penalty'
                ])
                ->where('status', 'completed')
                ->whereBetween('created_at', [$from, $to])
                ->sum('amount')
        );
    }

    /**
     * محاسبه هزینه بر اساس نوع
     */
    private function getCostByType(User $user, $type, $from, $to): float
    {
        return abs(
            $user->walletTransactions()
                ->where('type', $type)
                ->where('status', 'completed')
                ->whereBetween('created_at', [$from, $to])
                ->sum('amount')
        );
    }

    /**
     * بررسی موجودی کافی
     */
    public function hasEnoughBalance(User $user, float $amount): bool
    {
        $balance = $user->wallet_balance ? $user->wallet_balance : 0;
        return $balance >= $amount;
    }

    /**
     * دریافت موجودی فرمت شده
     */
    public function getFormattedBalance(User $user): string
    {
        $balance = $user->wallet_balance ? $user->wallet_balance : 0;
        return number_format($balance) . ' تومان';
    }

    /**
     * دریافت آخرین تراکنش
     */
    public function getLastTransaction(User $user)
    {
        return $user->walletTransactions()
            ->orderBy('created_at', 'desc')
            ->first();
    }

    /**
     * لغو تراکنش (در صورت نیاز)
     */
    public function cancelTransaction(WalletTransaction $transaction)
    {
        if ($transaction->status !== 'pending') {
            return false;
        }

        return DB::transaction(function () use ($transaction) {
            $transaction->status = 'cancelled';
            $transaction->save();

            Log::info('🚫 Transaction Cancelled', [
                'transaction_id' => $transaction->id,
                'user_id' => $transaction->user_id,
                'amount' => $transaction->amount,
            ]);

            return true;
        });
    }
}