// Primary IPs Management JavaScript
(function() {
    'use strict';

    // ==================== تنظیمات اولیه ====================
    const projectIdElement = document.querySelector('[data-project-id]');
    const projectId = projectIdElement ? projectIdElement.dataset.projectId : null;

    if (!projectId) {
        console.error('❌ Project ID not found! Add data-project-id attribute to your HTML element.');
        return;
    }

    const baseUrl = `/cloud/projects/${projectId}/servers/primaryips`;
    const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content');

    console.log('✅ Primary IPs Manager initialized for project:', projectId);

    // ==================== دکمه ایجاد Primary IP ====================
    const btnCreatePrimaryIp = document.getElementById('btnCreatePrimaryIp');
    if (btnCreatePrimaryIp) {
        btnCreatePrimaryIp.addEventListener('click', function() {
            console.log('🔵 Opening Create Primary IP modal');
            const modal = new bootstrap.Modal(document.getElementById('createPrimaryIpModal'));
            modal.show();
        });
    }

    // ==================== Auto Assign Checkbox ====================
const autoAssignCheckbox = document.getElementById('autoAssign');
if (autoAssignCheckbox) {
    autoAssignCheckbox.addEventListener('change', function() {
        const serverSelectBox = document.getElementById('serverSelectBox');
        if (this.checked) {
            serverSelectBox.classList.remove('d-none');
        } else {
            serverSelectBox.classList.add('d-none');
        }
    });
}

    // ==================== فیلتر Location در انتخاب سرور (Create Modal) ====================
    const locationSelect = document.querySelector('#createPrimaryIpModal select[name="location"]');
    if (locationSelect) {
        locationSelect.addEventListener('change', function() {
            const selectedLocation = this.value;
            const serverSelect = document.getElementById('serverSelect');
            if (serverSelect) {
                const options = serverSelect.querySelectorAll('option');
                
                options.forEach(option => {
                    if (option.value === '') return;
                    const serverLocation = option.dataset.location?.toLowerCase();
                    if (serverLocation === selectedLocation.toLowerCase()) {
                        option.style.display = 'block';
                    } else {
                        option.style.display = 'none';
                    }
                });
            }
        });
    }

    // ==================== فرم ایجاد Primary IP ====================
    const createPrimaryIpForm = document.getElementById('createPrimaryIpForm');
    if (createPrimaryIpForm) {
        createPrimaryIpForm.addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const btn = this.querySelector('button[type="submit"]');
            const originalText = btn.innerHTML;
            btn.disabled = true;
            btn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> در حال ایجاد...';
            
            const formData = new FormData(this);
            const data = Object.fromEntries(formData.entries());
            data.auto_assign = formData.has('auto_assign');
            
            console.log('📤 Creating Primary IP:', data);
            
            try {
                const response = await fetch(baseUrl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': csrfToken,
                        'Accept': 'application/json'
                    },
                    body: JSON.stringify(data)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    bootstrap.Modal.getInstance(document.getElementById('createPrimaryIpModal')).hide();
                    await Swal.fire({
                        icon: 'success',
                        title: '✅ Primary IP ساخته شد',
                        html: `
                            <p>IP: <code>${result.ip.ip}</code></p>
                            <p>Location: <strong>${result.ip.location}</strong></p>
                            <p class="text-danger">هزینه: ${result.ip.hourly_cost} تومان/ساعت</p>
                        `,
                        timer: 3000
                    });
                    location.reload();
                } else {
                    throw new Error(result.message || 'خطا در ایجاد Primary IP');
                }
            } catch (error) {
                console.error('❌ Create IP failed:', error);
                Swal.fire('خطا', error.message, 'error');
            } finally {
                btn.disabled = false;
                btn.innerHTML = originalText;
            }
        });
    }

    // ==================== Assign to Server ====================
    function attachAssignListeners() {
        document.querySelectorAll('.btn-assign-ip').forEach(btn => {
            btn.addEventListener('click', async function(e) {
                e.preventDefault();
                
                const ipId = this.dataset.ipId;
                const ipAddress = this.dataset.ipAddress || 'نامشخص';
                const location = this.dataset.location || 'نامشخص';
                
                console.log('🔵 Opening assign modal for IP:', { ipId, ipAddress, location });
                
                // پر کردن فیلدهای مدال
                document.getElementById('assignIpId').value = ipId;
                document.getElementById('assignIpAddress').value = ipAddress;
                document.getElementById('assignIpLocation').value = location;
                
                // نمایش مدال
                const modal = new bootstrap.Modal(document.getElementById('assignIpModal'));
                modal.show();
                
                // دریافت لیست سرورها
                const serverSelect = document.getElementById('assignServerSelect');
                serverSelect.innerHTML = '<option value="">در حال بارگذاری...</option>';
                
                try {
                    const response = await fetch(`${baseUrl}/${ipId}/available-servers`, {
                        method: 'GET',
                        headers: {
                            'X-Requested-With': 'XMLHttpRequest',
                            'Accept': 'application/json',
                            'X-CSRF-TOKEN': csrfToken
                        }
                    });
                    
                    const result = await response.json();
                    console.log('✅ Servers received:', result);
                    
                    serverSelect.innerHTML = '<option value="">انتخاب کنید</option>';
                    
                    if (!result.success) {
                        serverSelect.innerHTML = '<option value="" disabled>خطا: ' + result.message + '</option>';
                        return;
                    }
                    
                    if (!result.servers || result.servers.length === 0) {
                        serverSelect.innerHTML = `<option value="" disabled>سروری در Location ${result.ip_location} یافت نشد</option>`;
                        console.log('⚠️ No servers found in location:', result.ip_location);
                    } else {
                        console.log('✅ Found ' + result.servers.length + ' server(s)');
                        
                        result.servers.forEach(function(server) {
                            const option = document.createElement('option');
                            option.value = server.id;
                            
                            if (server.disabled) {
                                option.text = `${server.name} - ${server.disabled_reason}`;
                                option.disabled = true;
                                option.classList.add('text-muted');
                            } else if (!['stopped', 'off'].includes(server.status)) {
                                option.text = `${server.name} (${server.region}) - ⚠️ باید خاموش باشد`;
                                option.disabled = true;
                                option.classList.add('text-muted');
                            } else {
                                option.text = `${server.name} (${server.region}) - ${server.status}`;
                            }
                            
                            serverSelect.appendChild(option);
                        });
                    }
                    
                } catch (error) {
                    console.error('❌ Error fetching servers:', error);
                    serverSelect.innerHTML = '<option value="" disabled>خطا در دریافت لیست سرورها</option>';
                    Swal.fire('خطا', 'خطا در دریافت لیست سرورها: ' + error.message, 'error');
                }
            });
        });
    }

    // اجرای اولیه
    attachAssignListeners();

    // فرم Assign
    const assignIpForm = document.getElementById('assignIpForm');
    if (assignIpForm) {
        assignIpForm.addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const ipId = document.getElementById('assignIpId').value;
            const serverId = document.getElementById('assignServerSelect').value;
            
            console.log('📤 Submitting assign:', { ipId, serverId, projectId });
            
            if (!serverId) {
                Swal.fire('خطا', 'لطفاً سرور را انتخاب کنید', 'error');
                return;
            }
            
            const btn = this.querySelector('button[type="submit"]');
            const originalText = btn.innerHTML;
            btn.disabled = true;
            btn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> در حال اتصال...';
            
            try {
                const response = await fetch(`${baseUrl}/${ipId}/assign`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': csrfToken,
                        'Accept': 'application/json'
                    },
                    body: JSON.stringify({ server_id: serverId })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    console.log('✅ Assign successful:', result);
                    bootstrap.Modal.getInstance(document.getElementById('assignIpModal')).hide();
                    await Swal.fire({
                        icon: 'success',
                        title: 'موفق',
                        text: result.message || 'IP با موفقیت به سرور متصل شد',
                        timer: 2000,
                        showConfirmButton: false
                    });
                    location.reload();
                } else {
                    throw new Error(result.message || 'خطا در اتصال IP');
                }
            } catch (error) {
                console.error('❌ Assign failed:', error);
                Swal.fire('خطا', error.message, 'error');
            } finally {
                btn.disabled = false;
                btn.innerHTML = originalText;
            }
        });
    }

    // ==================== Unassign from Server ====================
    function attachUnassignListeners() {
        document.querySelectorAll('.btn-unassign-ip').forEach(btn => {
            btn.addEventListener('click', function(e) {
                e.preventDefault();
                
                const ipId = this.dataset.ipId;
                const ipAddress = this.dataset.ipAddress;
                
                console.log('🟡 Opening unassign modal for IP:', { ipId, ipAddress });
                
                document.getElementById('unassignIpId').value = ipId;
                document.getElementById('unassignIpAddress').value = ipAddress;
                
                const modal = new bootstrap.Modal(document.getElementById('unassignIpModal'));
                modal.show();
            });
        });
    }

    attachUnassignListeners();

    // فرم Unassign
    const unassignIpForm = document.getElementById('unassignIpForm');
    if (unassignIpForm) {
        unassignIpForm.addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const ipId = document.getElementById('unassignIpId').value;
            const btn = this.querySelector('button[type="submit"]');
            const originalText = btn.innerHTML;
            btn.disabled = true;
            btn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> در حال جداسازی...';
            
            console.log('📤 Unassigning IP:', ipId);
            
            try {
                const response = await fetch(`${baseUrl}/${ipId}/unassign`, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': csrfToken,
                        'Accept': 'application/json'
                    }
                });
                
                const result = await response.json();
                
                if (result.success) {
                    console.log('✅ Unassign successful');
                    bootstrap.Modal.getInstance(document.getElementById('unassignIpModal')).hide();
                    await Swal.fire('موفق', result.message || 'IP از سرور جدا شد', 'success');
                    location.reload();
                } else {
                    throw new Error(result.message || 'خطا در جداسازی IP');
                }
            } catch (error) {
                console.error('❌ Unassign failed:', error);
                Swal.fire('خطا', error.message, 'error');
            } finally {
                btn.disabled = false;
                btn.innerHTML = originalText;
            }
        });
    }

    // ==================== Edit RDNS ====================
    function attachRdnsListeners() {
        document.querySelectorAll('.btn-edit-rdns').forEach(btn => {
            btn.addEventListener('click', function(e) {
                e.preventDefault();
                
                const ipId = this.dataset.ipId;
                const ipAddress = this.dataset.ipAddress;
                const rdns = this.dataset.rdns;
                
                console.log('🔵 Opening RDNS modal for IP:', { ipId, ipAddress, rdns });
                
                document.getElementById('rdnsIpId').value = ipId;
                document.getElementById('rdnsIpAddress').value = ipAddress;
                document.getElementById('rdnsValue').value = rdns || '';
                
                const modal = new bootstrap.Modal(document.getElementById('editRdnsModal'));
                modal.show();
            });
        });
    }

    attachRdnsListeners();

    // فرم Edit RDNS
    const editRdnsForm = document.getElementById('editRdnsForm');
    if (editRdnsForm) {
        editRdnsForm.addEventListener('submit', async function(e) {
            e.preventDefault();
            
            const ipId = document.getElementById('rdnsIpId').value;
            const rdns = document.getElementById('rdnsValue').value;
            const btn = this.querySelector('button[type="submit"]');
            const originalText = btn.innerHTML;
            btn.disabled = true;
            btn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> در حال به‌روزرسانی...';
            
            console.log('📤 Updating RDNS:', { ipId, rdns });
            
            try {
                const response = await fetch(`${baseUrl}/${ipId}/rdns`, {
                    method: 'PUT',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': csrfToken,
                        'Accept': 'application/json'
                    },
                    body: JSON.stringify({ rdns })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    console.log('✅ RDNS updated');
                    bootstrap.Modal.getInstance(document.getElementById('editRdnsModal')).hide();
                    await Swal.fire('موفق', 'Reverse DNS به‌روزرسانی شد', 'success');
                    location.reload();
                } else {
                    throw new Error(result.message || 'خطا در به‌روزرسانی RDNS');
                }
            } catch (error) {
                console.error('❌ RDNS update failed:', error);
                Swal.fire('خطا', error.message, 'error');
            } finally {
                btn.disabled = false;
                btn.innerHTML = originalText;
            }
        });
    }

// ==================== Delete IP ====================
function attachDeleteListeners() {
    document.querySelectorAll('.btn-delete-ip').forEach(btn => {
        btn.addEventListener('click', function(e) {
            e.preventDefault();
            
            const ipId = this.dataset.ipId;
            const ipAddress = this.dataset.ipAddress;
            const cost = this.dataset.cost;
            
            console.log('🔴 Opening delete modal for IP:', { ipId, ipAddress, cost });
            
            document.getElementById('deleteIpId').value = ipId;
            document.getElementById('deleteIpAddress').value = ipAddress;
            
            const costBox = document.getElementById('deleteCostBox');
            if (cost && cost !== '0') {
                document.getElementById('deleteIpCost').value = cost + ' تومان';
                costBox.style.display = 'block';
            } else {
                costBox.style.display = 'none';
            }
            
            // ✅ ریست input تایید و دکمه
            const confirmInput = document.getElementById('confirmDeleteInput');
            if (confirmInput) {
                confirmInput.value = '';
            }
            
            const deleteBtn = document.getElementById('confirmDeleteBtn');
            deleteBtn.disabled = true;
            deleteBtn.classList.remove('btn-danger');
            deleteBtn.classList.add('btn-secondary');
            
            const modal = new bootstrap.Modal(document.getElementById('deleteIpModal'));
            modal.show();
        });
    });
}

attachDeleteListeners();

// ==================== Enable/Disable Delete Button ====================
const confirmDeleteInput = document.getElementById('confirmDeleteInput');
if (confirmDeleteInput) {
    confirmDeleteInput.addEventListener('input', function() {
        const deleteBtn = document.getElementById('confirmDeleteBtn');
        
        // ✅ فقط وقتی دقیقاً "delete me" نوشته شد، دکمه فعال میشه
        if (this.value === 'delete me') {
            deleteBtn.disabled = false;
            deleteBtn.classList.remove('btn-secondary');
            deleteBtn.classList.add('btn-danger');
        } else {
            deleteBtn.disabled = true;
            deleteBtn.classList.remove('btn-danger');
            deleteBtn.classList.add('btn-secondary');
        }
    });
}

// ==================== فرم Delete ====================
const deleteIpForm = document.getElementById('deleteIpForm');
if (deleteIpForm) {
    deleteIpForm.addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const ipId = document.getElementById('deleteIpId').value;
        const confirmInput = document.getElementById('confirmDeleteInput');
        
        // ✅ تایید نهایی قبل از حذف
        if (!confirmInput || confirmInput.value !== 'delete me') {
            Swal.fire('خطا', 'لطفاً عبارت "delete me" را وارد کنید', 'error');
            return;
        }
        
        const btn = document.getElementById('confirmDeleteBtn');
        const originalText = btn.innerHTML;
        btn.disabled = true;
        btn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> در حال حذف...';
        
        console.log('📤 Deleting IP:', ipId);
        
        try {
            const response = await fetch(`${baseUrl}/${ipId}`, {
                method: 'DELETE',
                headers: {
                    'X-CSRF-TOKEN': csrfToken,
                    'Accept': 'application/json'
                }
            });
            
            const result = await response.json();
            
            if (result.success) {
                console.log('✅ IP deleted');
                bootstrap.Modal.getInstance(document.getElementById('deleteIpModal')).hide();
                await Swal.fire({
                    icon: 'success',
                    title: 'موفق',
                    text: 'Primary IP حذف شد',
                    timer: 2000,
                    showConfirmButton: false
                });
                location.reload();
            } else {
                throw new Error(result.message || 'خطا در حذف IP');
            }
        } catch (error) {
            console.error('❌ Delete failed:', error);
            Swal.fire('خطا', error.message, 'error');
            
            // ✅ بازگشت به حالت اولیه
            btn.disabled = false;
            btn.innerHTML = originalText;
            btn.classList.remove('btn-danger');
            btn.classList.add('btn-secondary');
        }
    });
}

    // ==================== Filters ====================
    const btnFilter = document.getElementById('btnFilter');
    if (btnFilter) {
        btnFilter.addEventListener('click', function() {
            const filterBox = document.getElementById('filterBox');
            if (filterBox) {
                filterBox.classList.toggle('d-none');
            }
        });
    }

    // Apply Filters
    function applyFilters() {
        const type = document.getElementById('filterType')?.value || '';
        const status = document.getElementById('filterStatus')?.value || '';
        const location = document.getElementById('filterLocation')?.value || '';
        
        const rows = document.querySelectorAll('#primaryIpsTable tbody tr[data-ip-id]');
        
        rows.forEach(row => {
            let show = true;
            
            if (type && row.dataset.type !== type) show = false;
            if (status && row.dataset.status !== status) show = false;
            if (location && row.dataset.location !== location) show = false;
            
            row.style.display = show ? '' : 'none';
        });
    }

    ['filterType', 'filterStatus', 'filterLocation'].forEach(id => {
        const element = document.getElementById(id);
        if (element) {
            element.addEventListener('change', applyFilters);
        }
    });

    const btnResetFilter = document.getElementById('btnResetFilter');
    if (btnResetFilter) {
        btnResetFilter.addEventListener('click', function() {
            const filterType = document.getElementById('filterType');
            const filterStatus = document.getElementById('filterStatus');
            const filterLocation = document.getElementById('filterLocation');
            
            if (filterType) filterType.value = '';
            if (filterStatus) filterStatus.value = '';
            if (filterLocation) filterLocation.value = '';
            
            applyFilters();
        });
    }

    // ==================== Select All ====================
    const selectAllCheckbox = document.getElementById('selectAll');
    if (selectAllCheckbox) {
        selectAllCheckbox.addEventListener('change', function() {
            document.querySelectorAll('.row-select').forEach(cb => {
                cb.checked = this.checked;
            });
        });
    }

    console.log('✅ All event listeners attached successfully');

})();