@php
    $user = auth()->user();
    $remainingChanges = $user->getRemainingIpChanges();
    $usedChanges = 5 - $remainingChanges;
    $changePercentage = ($usedChanges / 5) * 100;
@endphp

<div class="card" dir="rtl">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="mb-0">
            <i class="ri-global-line me-2"></i>
            مدیریت IP ها
        </h5>
        
        <div>
            @if($server->provider === 'hetzner' || $server->provider === 'vultr')
            <button class="btn btn-sm btn-primary" id="btnAddIp">
                <i class="ri-add-line me-1"></i>
                افزودن Floating IP
            </button>
            @endif
        </div>
    </div>

    <div class="card-body">
        
        {{-- محدودیت تغییر IP ماهانه (فقط Hetzner) --}}
        @if($server->provider === 'hetzner')
        <div class="alert alert-info mb-4">
            <div class="d-flex justify-content-between align-items-center mb-2">
                <span>
                    <i class="ri-refresh-line me-2"></i>
                    <strong>تغییر Primary IP این ماه:</strong> 
                    {{ $usedChanges }} از 5 بار
                    <small class="text-muted">(هزینه هر بار: 3,700 تومان)</small>
                </span>
                <span class="badge bg-primary">{{ $remainingChanges }} بار باقی‌مانده</span>
            </div>
            <div class="progress" style="height: 8px;">
                <div class="progress-bar bg-{{ $changePercentage > 80 ? 'danger' : ($changePercentage > 60 ? 'warning' : 'success') }}" 
                     role="progressbar" 
                     style="width: {{ $changePercentage }}%"
                     aria-valuenow="{{ $usedChanges }}" 
                     aria-valuemin="0" 
                     aria-valuemax="5">
                </div>
            </div>
            <small class="text-muted mt-2 d-block">
                <i class="ri-information-line"></i>
                محدودیت تغییر IP در ماه جاری: {{ $usedChanges }}/5 | اول ماه آینده ریست می‌شود
            </small>
        </div>
        @endif

        {{-- وضعیت سرور --}}
        @if($server->provider === 'hetzner' && !in_array($server->status, ['stopped', 'off']))
        <div class="alert alert-warning">
            <i class="ri-alert-line me-2"></i>
            برای افزودن، حذف یا تغییر IP، سرور باید خاموش باشد.
            <button class="btn btn-sm btn-warning ms-2" onclick="powerOff()">
                <i class="ri-shut-down-line me-1"></i>
                خاموش کردن سرور
            </button>
        </div>
        @endif

{{-- IP های اصلی --}}
<h6 class="mb-3 mt-4">
    <i class="ri-shield-star-line me-2 text-primary"></i>
    IP های اصلی (Primary IPs)
    @if($server->provider === 'hetzner')
    <small class="text-muted">(IPv4: 250 تومان/ساعت | IPv6: 120 تومان/ساعت)</small>
    @elseif($server->provider === 'vultr')
    <small class="text-muted">(رایگان - شامل قیمت سرور)</small>
    @else
    <small class="text-muted">(شامل قیمت پلن)</small>
    @endif
</h6>

<div class="table-responsive mb-4">
    <table class="table table-hover align-middle">
        <thead class="table-light">
            <tr>
                <th style="width: 25%">آی‌پی</th>
                <th style="width: 12%">نوع</th>
                @if($server->provider === 'hetzner' || $server->provider === 'vultr')
                <th style="width: 15%">Netmask</th>
                <th style="width: 15%">Gateway</th>
                @endif
                <th style="width: 15%">RDNS</th>
                <th style="width: 13%">هزینه</th>
                <th style="width: 5%">عملیات</th>
            </tr>
        </thead>
        <tbody id="primaryIpsTable">
            @forelse($mainIps as $ip)
            <tr>
                <td>
                    <code class="bg-light px-2 py-1 rounded">{{ $ip['ip'] }}</code>
                </td>
                <td>
                    <span class="badge bg-primary">{{ strtoupper($ip['type']) }}</span>
                    <span class="badge bg-success">Primary</span>
                </td>
                @if($server->provider === 'hetzner' || $server->provider === 'vultr')
                <td>
                    @if(isset($ip['netmask']) && $ip['netmask'])
                    <code class="text-muted small">{{ $ip['netmask'] }}</code>
                    @else
                    <span class="text-muted">—</span>
                    @endif
                </td>
                <td>
                    @if(isset($ip['gateway']) && $ip['gateway'])
                    <code class="text-muted small">{{ $ip['gateway'] }}</code>
                    @else
                    <span class="text-muted">—</span>
                    @endif
                </td>
                @endif
                <td>
                    <small class="text-muted">{{ $ip['rdns'] ?? 'تنظیم نشده' }}</small>
                </td>
                <td>
                    @if($server->provider === 'hetzner')
                        @if($ip['type'] === 'ipv4')
                        <small class="text-danger">
                            <i class="ri-money-dollar-circle-line me-1"></i>
                            250 ت/س
                        </small>
                        @else
                        <small class="text-warning">
                        <i class="ri-gift-line me-1"></i>
                        رایگان
                    </span>
                        @endif
                    @else
                    <span class="text-success fw-bold">
                        <i class="ri-gift-line me-1"></i>
                        رایگان
                    </span>
                    @endif
                </td>
                <td>
                    <div class="btn-group btn-group-sm">
                        @if($server->provider === 'hetzner' && $ip['type'] === 'ipv4')
                        <button class="btn btn-warning btn-change-primary" 
                                data-ip-id="{{ $ip['id'] }}"
                                data-ip-address="{{ $ip['ip'] }}"
                                title="تغییر Primary IP (3,700 تومان)"
                                @if(!in_array($server->status, ['stopped', 'off']) || $remainingChanges <= 0) disabled @endif>
                            <i class="ri-refresh-line"></i>
                        </button>
                        @endif
                        
                        <button class="btn btn-outline-secondary btn-edit-rdns" 
                                data-ip-id="{{ $ip['id'] }}"
                                data-ip-address="{{ $ip['ip'] }}"
                                title="ویرایش RDNS">
                            <i class="ri-edit-line"></i>
                        </button>
                    </div>
                </td>
            </tr>
            @empty
            <tr>
                <td colspan="{{ ($server->provider === 'hetzner' || $server->provider === 'vultr') ? '7' : '5' }}" class="text-center text-muted py-4">
                    <i class="ri-information-line fs-3 d-block mb-2"></i>
                    IP اصلی یافت نشد
                </td>
            </tr>
            @endforelse
        </tbody>
    </table>
</div>

{{-- ✅ دکمه فعال‌سازی IPv6 برای Vultr --}}
@if($server->provider === 'vultr')
    @php
        // چک کردن وجود IPv6
        $hasIPv6 = collect($mainIps)->contains(function($ip) {
            return $ip['type'] === 'ipv6';
        });
    @endphp
    
    @if(!$hasIPv6)
    <div class="alert alert-info mb-4">
        <div class="d-flex justify-content-between align-items-center">
            <div>
                <i class="ri-information-line me-2 fs-5"></i>
                <strong>IPv6 فعال نیست</strong>
                <p class="mb-0 mt-1 text-muted small">
                    می‌توانید IPv6 را به صورت رایگان برای این سرور فعال کنید.
                </p>
            </div>
            <button class="btn btn-primary" id="btnEnableIPv6" style="min-width: 150px;">
                <i class="ri-add-line me-1"></i>
                فعال‌سازی IPv6
            </button>
        </div>
    </div>
    @else
    <div class="alert alert-success mb-4">
        <i class="ri-checkbox-circle-line me-2"></i>
        <strong>IPv6 فعال است</strong>
        <span class="text-muted">— این سرور دارای IPv6 رایگان است</span>
    </div>
    @endif
@endif

        {{-- IP های اضافه (Floating) --}}
        @if($server->provider === 'hetzner' || $server->provider === 'vultr')
        <h6 class="mb-3 mt-5">
            <i class="ri-add-circle-line me-2 text-info"></i>
            IP های اضافه (Floating IPs)
            @if($server->provider === 'hetzner')
            <small class="text-muted">(IPv4: 750,000 تومان/ماه | IPv6: 300,000 تومان/ماه)</small>
            @endif
        </h6>
        
        <div class="table-responsive">
            <table class="table table-hover align-middle">
                <thead class="table-light">
                    <tr>
                        <th style="width: 25%">آی‌پی</th>
                        <th style="width: 12%">نوع</th>
                        <th style="width: 18%">RDNS</th>
                        <th style="width: 20%">هزینه</th>
                        <th style="width: 10%">تاریخ ایجاد</th>
                        <th style="width: 15%">عملیات</th>
                    </tr>
                </thead>
                <tbody id="floatingIpsTable">
                    @forelse($additionalIps as $ip)
                    @php
                        if ($server->provider === 'hetzner') {
                            $monthlyPrice = $ip['type'] === 'ipv6' ? 300000 : 750000;
                        } else {
                            $monthlyPrice = 0;
                        }
                        $createdAt = $ip['created_at'] ?? null;
                    @endphp
                    <tr data-ip-id="{{ $ip['id'] }}">
                        <td>
                            <code class="bg-light px-2 py-1 rounded">{{ $ip['ip'] }}</code>
                        </td>
                        <td>
                            <span class="badge bg-secondary">{{ strtoupper($ip['type']) }}</span>
                            <span class="badge bg-warning text-dark">Floating</span>
                        </td>
                        <td>
                            <small class="text-muted">{{ $ip['rdns'] ?? 'تنظیم نشده' }}</small>
                        </td>
                        <td>
                            @if($monthlyPrice > 0)
                            <small class="text-danger">
                                <i class="ri-money-dollar-circle-line me-1"></i>
                                {{ number_format($monthlyPrice) }} ت/ماه
                            </small>
                            @else
                            <span class="text-success">
                                <i class="ri-gift-line me-1"></i>
                                رایگان
                            </span>
                            @endif
                        </td>
                        <td>
                            @if($createdAt)
                            <small class="text-muted">
                                {{ \Carbon\Carbon::parse($createdAt)->format('Y/m/d') }}
                            </small>
                            @else
                            <small class="text-muted">—</small>
                            @endif
                        </td>
                        <td>
                            <button class="btn btn-sm btn-danger btn-delete-ip" 
                                    data-ip-id="{{ $ip['id'] }}"
                                    data-ip-address="{{ $ip['ip'] }}"
                                    data-ip-type="{{ $ip['type'] }}"
                                    data-monthly-price="{{ $monthlyPrice }}"
                                    @if($server->provider === 'hetzner' && !in_array($server->status, ['stopped', 'off'])) disabled @endif>
                                <i class="ri-delete-bin-line me-1"></i>
                                حذف
                            </button>
                        </td>
                    </tr>
                    @empty
                    <tr>
                        <td colspan="6" class="text-center text-muted py-5">
                            <i class="ri-inbox-line fs-1 d-block mb-3 text-secondary"></i>
                            <p class="mb-2">Floating IP وجود ندارد</p>
                            <small>با کلیک روی "افزودن Floating IP" می‌توانید یک IP اضافه کنید</small>
                        </td>
                    </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
        @endif

        {{-- راهنمای استفاده --}}
        <div class="card bg-light mt-4">
            <div class="card-body">
                <h6 class="card-title">
                    <i class="ri-information-line me-2"></i>
                    راهنمای استفاده
                </h6>
                <ul class="mb-0 small">
                    @if($server->provider === 'hetzner')
                    <li><strong>Primary IPv4:</strong> 250 تومان/ساعت (~182,500 تومان/ماه) - کسر ساعتی</li>
                    <li><strong>Primary IPv6:</strong> 120 تومان/ساعت (~87,600 تومان/ماه) - کسر ساعتی</li>
                    <li><strong>Floating IPv4:</strong> 750,000 تومان/ماه - کسر در اول ماه</li>
                    <li><strong>Floating IPv6:</strong> 300,000 تومان/ماه - کسر در اول ماه</li>
                    <li><strong>تغییر Primary IP:</strong> 3,700 تومان (محدودیت: 5 بار/ماه)</li>
                    <li><strong>حذف Floating IP:</strong> بدون جریمه</li>
                    <li><strong>⚙️ شرایط:</strong> سرور باید خاموش باشد</li>
                    @elseif($server->provider === 'vultr')
                    <li><strong>Primary IPv4:</strong> رایگان (شامل قیمت سرور)</li>
                    <li><strong>Primary IPv6:</strong> رایگان (شامل قیمت سرور)</li>
                    <li><strong>Netmask & Gateway:</strong> نمایش داده می‌شود</li>
                    <li><strong>Additional IPv4:</strong> امکان افزودن وجود ندارد</li>
                    @endif
                </ul>
            </div>
        </div>

    </div>
</div>

{{-- Modals --}}
<div class="modal fade" id="addIpModal" tabindex="-1" dir="rtl">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="ri-add-line me-2"></i>
                    افزودن Floating IP جدید
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                
                @if($server->provider === 'hetzner')
                <div class="alert alert-info">
                    <i class="ri-information-line me-2"></i>
                    <strong>هتزنر:</strong> می‌توانید Floating IP اضافه کنید.
                    <br>
                    <small>⚠️ سرور باید خاموش باشد | هزینه بلافاصله کسر می‌شود</small>
                </div>
                @elseif($server->provider === 'vultr')
                <div class="alert alert-info">
                    <i class="ri-information-line me-2"></i>
                    <strong>ولتر:</strong> فقط IPv6 قابل افزودن است (رایگان).
                </div>
                @endif

                <div class="mb-3">
                    <label class="form-label">نوع IP</label>
                    <select class="form-select" id="ipType">
                        @if($server->provider === 'hetzner')
                        <option value="ipv4">IPv4 (750,000 تومان/ماه)</option>
                        <option value="ipv6">IPv6 (300,000 تومان/ماه)</option>
                        @else
                        <option value="ipv6">IPv6 (رایگان)</option>
                        @endif
                    </select>
                </div>

                @if($server->provider === 'hetzner')
                <div class="alert alert-success">
                    <i class="ri-checkbox-circle-line me-2"></i>
                    <strong>✅ حذف بدون جریمه</strong>
                    <p class="mb-0 small">Floating IP ها ماهانه هستند و حذف آن‌ها جریمه ندارد</p>
                </div>
                @endif

                <div class="card bg-light">
                    <div class="card-body">
                        <h6 class="card-title">خلاصه هزینه</h6>
                        <table class="table table-sm table-borderless mb-0">
                            <tr>
                                <td>هزینه ماهانه:</td>
                                <td class="text-end"><strong id="monthlyCost">—</strong></td>
                            </tr>
                            <tr>
                                <td>نحوه پرداخت:</td>
                                <td class="text-end"><span class="badge bg-info" id="paymentType">—</span></td>
                            </tr>
                            <tr class="border-top">
                                <td>جریمه حذف:</td>
                                <td class="text-end text-success"><strong>ندارد ✓</strong></td>
                            </tr>
                        </table>
                    </div>
                </div>

            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">انصراف</button>
                <button type="button" class="btn btn-primary" id="confirmAddIpBtn">
                    <i class="ri-add-line me-1"></i>
                    افزودن IP
                </button>
            </div>
        </div>
    </div>
</div>

<div class="modal fade" id="changeIpModal" tabindex="-1" dir="rtl">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="ri-refresh-line me-2"></i>
                    تغییر Primary IP
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                
                <div class="alert alert-warning">
                    <i class="ri-alert-line me-2"></i>
                    <strong>هزینه تغییر:</strong> 3,700 تومان
                    <br>
                    <small>باقی‌مانده این ماه: {{ $remainingChanges }} از 5 بار</small>
                </div>

                <div class="mb-3">
                    <label class="form-label">IP فعلی</label>
                    <input type="text" class="form-control" id="currentIp" readonly>
                </div>

                <div class="mb-3">
                    <label class="form-label">انتخاب IP جدید</label>
                    <select class="form-select" id="newIpSelect">
                        <option value="">ایجاد IP جدید</option>
                    </select>
                    <small class="text-muted">می‌توانید IP جدید ایجاد کنید یا از IP های قبلی استفاده کنید</small>
                </div>

            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">انصراف</button>
                <button type="button" class="btn btn-warning" id="confirmChangeIpBtn">
                    <i class="ri-refresh-line me-1"></i>
                    تأیید تغییر (3,700 تومان)
                </button>
            </div>
        </div>
    </div>
</div>

<div class="modal fade" id="rdnsModal" tabindex="-1" dir="rtl">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">ویرایش RDNS</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="mb-3">
                    <label class="form-label">IP Address</label>
                    <input type="text" class="form-control" id="rdnsIp" readonly>
                </div>
                <div class="mb-3">
                    <label class="form-label">RDNS (Reverse DNS)</label>
                    <input type="text" class="form-control" id="rdnsValue" placeholder="example.com">
                    <small class="text-muted">مثال: server.example.com</small>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">انصراف</button>
                <button type="button" class="btn btn-primary" id="confirmUpdateRdnsBtn">
                    <i class="ri-save-line me-1"></i>
                    ذخیره
                </button>
            </div>
        </div>
    </div>
</div>

<style>
.table code {
    font-size: 0.875rem;
}

.btn-group-sm .btn {
    padding: 0.25rem 0.5rem;
}

.progress {
    background-color: #e9ecef;
}
</style>

<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
<script>
(function() {
    'use strict';
    
    // ✅ شناسه یکتا برای این سرور
    const NETWORK_TAB_KEY = 'networkTab_{{ $server->id }}';
    
    console.log('🔵 Network Tab Script Executed');
    
    // ✅ جلوگیری از اجرای مجدد
    if (window[NETWORK_TAB_KEY]) {
        console.warn('⚠️ Network Tab already initialized for server {{ $server->id }}');
        return;
    }
    window[NETWORK_TAB_KEY] = true;
    
    console.log('🟢 Network Tab Initializing...');
    
    // ✅ چک کردن نیازمندی‌ها
    if (typeof bootstrap === 'undefined') {
        console.error('❌ Bootstrap not loaded');
        return;
    }
    
    if (typeof Swal === 'undefined') {
        console.error('❌ SweetAlert2 not loaded');
        return;
    }
    
    const csrfToken = document.querySelector('meta[name="csrf-token"]')?.content;
    if (!csrfToken) {
        console.error('❌ CSRF token not found');
        return;
    }
    
    // ✅ متغیرهای سرور
    const serverId = {{ $server->id }};
    const serverProvider = '{{ $server->provider }}';
    const serverStatus = '{{ $server->status }}';
    const serverProjectId = {{ $server->project_id ?? 'null' }};
    const remainingChanges = {{ $remainingChanges }};
    
    console.log('📊 Server Config:', { serverId, serverProvider, serverStatus, serverProjectId });
    
    let currentChangeIpId = null;
    
    // ==================== Helper Functions ====================
    function updateAddIpSummary() {
        const ipType = document.getElementById('ipType')?.value;
        const monthlyCostEl = document.getElementById('monthlyCost');
        const paymentTypeEl = document.getElementById('paymentType');
        
        if (!monthlyCostEl || !paymentTypeEl) return;
        
        if (serverProvider === 'hetzner') {
            if (ipType === 'ipv4') {
                monthlyCostEl.textContent = '750,000 تومان';
                paymentTypeEl.textContent = 'اول ماه';
            } else {
                monthlyCostEl.textContent = '300,000 تومان';
                paymentTypeEl.textContent = 'اول ماه';
            }
        } else {
            monthlyCostEl.textContent = 'رایگان';
            paymentTypeEl.textContent = 'ندارد';
        }
    }
    
    // ==================== دکمه افزودن Floating IP ====================
    const btnAddIp = document.getElementById('btnAddIp');
    if (btnAddIp) {
        console.log('✅ Add IP button found');
        btnAddIp.addEventListener('click', function() {
            console.log('🔵 Add IP clicked');
            
            if (serverProvider === 'hetzner' && !['stopped', 'off'].includes(serverStatus)) {
                Swal.fire('خطا', 'سرور باید خاموش باشد', 'warning');
                return;
            }
            
            updateAddIpSummary();
            
            const modalEl = document.getElementById('addIpModal');
            if (modalEl) {
                const modal = new bootstrap.Modal(modalEl);
                modal.show();
            } else {
                console.error('❌ addIpModal not found');
            }
        });
    }
    
    // تغییر نوع IP
    const ipTypeSelect = document.getElementById('ipType');
    if (ipTypeSelect) {
        ipTypeSelect.addEventListener('change', updateAddIpSummary);
    }
    
    // ==================== تأیید افزودن Floating IP ====================
    const confirmAddIpBtn = document.getElementById('confirmAddIpBtn');
    if (confirmAddIpBtn) {
        confirmAddIpBtn.addEventListener('click', async function() {
            const type = document.getElementById('ipType')?.value;
            if (!type) {
                Swal.fire('خطا', 'لطفاً نوع IP را انتخاب کنید', 'error');
                return;
            }
            
            const btn = this;
            btn.disabled = true;
            btn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> در حال پردازش...';
            
            try {
                const response = await fetch(`/servers/${serverId}/ips`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': csrfToken,
                        'Accept': 'application/json'
                    },
                    body: JSON.stringify({ type })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    const modalEl = document.getElementById('addIpModal');
                    if (modalEl) {
                        const modal = bootstrap.Modal.getInstance(modalEl);
                        if (modal) modal.hide();
                    }
                    
                    await Swal.fire({
                        icon: 'success',
                        title: '✅ Floating IP اضافه شد',
                        html: `<p>IP: <code>${data.ip?.ip || 'جدید'}</code></p>`,
                        timer: 2000,
                        showConfirmButton: false
                    });
                    location.reload();
                } else {
                    throw new Error(data.message || 'خطا در افزودن IP');
                }
            } catch (error) {
                console.error('❌ Add IP Error:', error);
                Swal.fire('خطا', error.message, 'error');
            } finally {
                btn.disabled = false;
                btn.innerHTML = '<i class="ri-add-line me-1"></i> افزودن IP';
            }
        });
    }
    
    // ==================== حذف Floating IP ====================
    const deleteIpButtons = document.querySelectorAll('.btn-delete-ip');
    console.log(`✅ Found ${deleteIpButtons.length} Delete IP buttons`);
    
    deleteIpButtons.forEach(btn => {
        btn.addEventListener('click', async function() {
            const ipId = this.dataset.ipId;
            const ipAddress = this.dataset.ipAddress;
            const monthlyPrice = this.dataset.monthlyPrice;
            
            const result = await Swal.fire({
                title: 'حذف Floating IP',
                html: `
                    <p>IP: <code>${ipAddress}</code></p>
                    ${monthlyPrice > 0 ? `<p class="text-muted">هزینه ماهانه: ${Number(monthlyPrice).toLocaleString()} تومان</p>` : ''}
                    <p class="text-success">✅ بدون جریمه</p>
                `,
                icon: 'warning',
                showCancelButton: true,
                confirmButtonText: 'حذف',
                confirmButtonColor: '#dc3545',
                cancelButtonText: 'انصراف'
            });
            
            if (!result.isConfirmed) return;
            
            try {
                const response = await fetch(`/servers/${serverId}/ips/${ipId}`, {
                    method: 'DELETE',
                    headers: { 
                        'X-CSRF-TOKEN': csrfToken, 
                        'Accept': 'application/json' 
                    }
                });
                
                const data = await response.json();
                if (data.success) {
                    await Swal.fire({
                        icon: 'success',
                        title: '✅ IP حذف شد',
                        timer: 2000,
                        showConfirmButton: false
                    });
                    location.reload();
                } else {
                    throw new Error(data.message || 'خطا در حذف IP');
                }
            } catch (error) {
                console.error('❌ Delete IP Error:', error);
                Swal.fire('خطا', error.message, 'error');
            }
        });
    });
    
    // ==================== تغییر Primary IP (فقط Hetzner) ====================
    const changePrimaryButtons = document.querySelectorAll('.btn-change-primary');
    console.log(`✅ Found ${changePrimaryButtons.length} Change Primary IP buttons`);
    
    changePrimaryButtons.forEach(btn => {
        btn.addEventListener('click', function() {
            console.log('🔵 Change IP button clicked');
            
            if (remainingChanges <= 0) {
                Swal.fire({
                    icon: 'warning',
                    title: 'محدودیت تغییر IP',
                    text: 'شما به حداکثر تعداد تغییر IP ماهانه رسیده‌اید (5 بار)'
                });
                return;
            }
            
            currentChangeIpId = this.dataset.ipId;
            const currentIpInput = document.getElementById('currentIp');
            if (currentIpInput) {
                currentIpInput.value = this.dataset.ipAddress;
            }
            
            const modalEl = document.getElementById('changeIpModal');
            if (modalEl) {
                const modal = new bootstrap.Modal(modalEl);
                modal.show();
            }
        });
    });
    
    // تأیید تغییر Primary IP
    const confirmChangeIpBtn = document.getElementById('confirmChangeIpBtn');
    if (confirmChangeIpBtn) {
        confirmChangeIpBtn.addEventListener('click', async function() {
            console.log('🔵 Confirm change IP clicked', currentChangeIpId);
            
            if (!currentChangeIpId) {
                Swal.fire('خطا', 'لطفاً یک IP انتخاب کنید', 'error');
                return;
            }
            
            const btn = this;
            btn.disabled = true;
            const originalText = btn.innerHTML;
            btn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> در حال پردازش...';
            
            try {
                const newIpId = document.getElementById('newIpSelect')?.value || null;
                
                const result = await Swal.fire({
                    title: 'تأیید تغییر Primary IP',
                    html: `
                        <div class="text-start">
                            <p><strong>هزینه:</strong> 3,700 تومان</p>
                            <p><strong>باقی‌مانده:</strong> ${remainingChanges} از 5 بار</p>
                            <div class="alert alert-warning mt-3">
                                <i class="ri-alert-line me-2"></i>
                                این عملیات غیرقابل بازگشت است
                            </div>
                        </div>
                    `,
                    icon: 'question',
                    showCancelButton: true,
                    confirmButtonText: 'تأیید و پرداخت',
                    cancelButtonText: 'انصراف',
                    confirmButtonColor: '#f59e0b'
                });
                
                if (!result.isConfirmed) {
                    btn.disabled = false;
                    btn.innerHTML = originalText;
                    return;
                }
                
                const response = await fetch(`/servers/${serverId}/ips/${currentChangeIpId}/change`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': csrfToken,
                        'Accept': 'application/json'
                    },
                    body: JSON.stringify({ new_ip_id: newIpId })
                });
                
                const data = await response.json();
                console.log('📥 Change IP Response:', data);
                
                if (data.success) {
                    const modalEl = document.getElementById('changeIpModal');
                    if (modalEl) {
                        const modal = bootstrap.Modal.getInstance(modalEl);
                        if (modal) modal.hide();
                    }
                    
                    await Swal.fire({
                        icon: 'success',
                        title: '✅ Primary IP تغییر کرد',
                        html: `
                            <div class="text-start">
                                <p><strong>IP جدید:</strong> <code>${data.new_ip?.ip || 'در حال اختصاص...'}</code></p>
                                <p><strong>هزینه کسر شده:</strong> 3,700 تومان</p>
                                <p><strong>تغییرات باقی‌مانده:</strong> ${data.remaining_changes || remainingChanges - 1} بار</p>
                            </div>
                        `,
                        timer: 4000,
                        showConfirmButton: true,
                        confirmButtonText: 'باشه'
                    });
                    
                    location.reload();
                } else {
                    throw new Error(data.message || 'خطا در تغییر IP');
                }
            } catch (error) {
                console.error('❌ Change IP Error:', error);
                Swal.fire({
                    icon: 'error',
                    title: 'خطا در تغییر IP',
                    html: `<p>${error.message}</p>`,
                });
            } finally {
                btn.disabled = false;
                btn.innerHTML = originalText;
            }
        });
    }
    
    // ==================== ویرایش RDNS ====================
    const rdnsButtons = document.querySelectorAll('.btn-edit-rdns');
    console.log(`✅ Found ${rdnsButtons.length} RDNS buttons`);
    
    rdnsButtons.forEach((btn, index) => {
        btn.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            const ipId = this.dataset.ipId;
            const ipAddress = this.dataset.ipAddress;
            
            console.log(`🔵 RDNS button ${index} clicked`, { ipId, ipAddress });
            
            // دریافت RDNS فعلی
            const row = this.closest('tr');
            let currentRdns = '';
            
            if (row) {
                @if($server->provider === 'vultr')
                const rdnsCell = row.querySelector('td:nth-child(5)');
                @else
                const rdnsCell = row.querySelector('td:nth-child(3)');
                @endif
                
                if (rdnsCell) {
                    currentRdns = rdnsCell.textContent.trim();
                    if (currentRdns === 'تنظیم نشده') {
                        currentRdns = '';
                    }
                }
            }
            
            // ست کردن مقادیر
            const rdnsIpInput = document.getElementById('rdnsIp');
            const rdnsValueInput = document.getElementById('rdnsValue');
            const rdnsIpIdInput = document.getElementById('rdnsIpId');
            
            if (rdnsIpInput) rdnsIpInput.value = ipAddress;
            if (rdnsValueInput) rdnsValueInput.value = currentRdns;
            if (rdnsIpIdInput) rdnsIpIdInput.value = ipId;
            
            console.log('✅ Set RDNS values:', { ipAddress, currentRdns, ipId });
            
            // باز کردن Modal
            const modalEl = document.getElementById('rdnsModal');
            if (modalEl) {
                console.log('🔓 Opening RDNS Modal...');
                try {
                    const modal = new bootstrap.Modal(modalEl, {
                        backdrop: 'static',
                        keyboard: true
                    });
                    modal.show();
                    console.log('✅ RDNS Modal opened');
                } catch (error) {
                    console.error('❌ Error opening RDNS modal:', error);
                    alert('خطا در باز کردن Modal: ' + error.message);
                }
            } else {
                console.error('❌ rdnsModal not found in DOM');
                alert('خطا: Modal RDNS یافت نشد');
            }
        });
    });
    
    // ==================== تأیید تنظیم RDNS ====================
    const confirmUpdateRdnsBtn = document.getElementById('confirmUpdateRdnsBtn');
    if (confirmUpdateRdnsBtn) {
        console.log('✅ Confirm RDNS button found');
        
        confirmUpdateRdnsBtn.addEventListener('click', async function() {
            const ipId = document.getElementById('rdnsIpId')?.value;
            const reverse = document.getElementById('rdnsValue')?.value.trim();
            
            console.log('🔵 Confirm RDNS', { ipId, reverse });
            
            if (!ipId) {
                Swal.fire('خطا', 'IP انتخاب نشده است', 'error');
                return;
            }
            
            if (!reverse) {
                Swal.fire('خطا', 'لطفاً Reverse DNS را وارد کنید', 'error');
                return;
            }
            
            // Validation
            const rdnsPattern = /^[a-zA-Z0-9\.\-]+$/;
            if (!rdnsPattern.test(reverse)) {
                Swal.fire('خطا', 'فرمت نامعتبر (فقط حروف، اعداد، نقطه و خط تیره)', 'error');
                return;
            }
            
            const btn = this;
            btn.disabled = true;
            btn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> در حال پردازش...';
            
            try {
                if (!serverProjectId) {
                    throw new Error('Project ID یافت نشد');
                }
                
                const url = `/cloud/projects/${serverProjectId}/servers/primaryips/${ipId}/rdns`;
                console.log('📤 Sending RDNS request to:', url);
                
                const response = await fetch(url, {
                    method: 'PUT',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': csrfToken,
                        'Accept': 'application/json'
                    },
                    body: JSON.stringify({ rdns: reverse })
                });
                
                const data = await response.json();
                console.log('📥 RDNS Response:', data);
                
                if (data.success) {
                    const modalEl = document.getElementById('rdnsModal');
                    if (modalEl) {
                        const modal = bootstrap.Modal.getInstance(modalEl);
                        if (modal) modal.hide();
                    }
                    
                    await Swal.fire({
                        icon: 'success',
                        title: '✅ RDNS تنظیم شد',
                        html: `<p>Reverse DNS: <code>${reverse}</code></p>`,
                        timer: 2000,
                        showConfirmButton: false
                    });
                    
                    location.reload();
                } else {
                    throw new Error(data.message || 'خطا در تنظیم RDNS');
                }
            } catch (error) {
                console.error('❌ RDNS Error:', error);
                Swal.fire('خطا', error.message, 'error');
            } finally {
                btn.disabled = false;
                btn.innerHTML = '<i class="ri-save-line me-1"></i> ذخیره';
            }
        });
    } else {
        console.warn('⚠️ confirmUpdateRdnsBtn not found');
    }
    
    // ==================== فعال‌سازی IPv6 (Vultr) ====================
    const enableIPv6Btn = document.getElementById('btnEnableIPv6');
    if (enableIPv6Btn) {
        console.log('✅ Enable IPv6 button found');
        
        enableIPv6Btn.addEventListener('click', async function() {
            console.log('🔵 Enable IPv6 clicked');
            
            const result = await Swal.fire({
                title: 'فعال‌سازی IPv6',
                html: `
                    <div class="text-start">
                        <p><i class="ri-gift-line text-success me-2"></i>IPv6 رایگان است</p>
                        <p><i class="ri-time-line text-info me-2"></i>ممکن است چند دقیقه طول بکشد</p>
                    </div>
                `,
                icon: 'question',
                showCancelButton: true,
                confirmButtonText: 'فعال‌سازی',
                cancelButtonText: 'انصراف',
            });
            
            if (!result.isConfirmed) return;
            
            const btn = this;
            btn.disabled = true;
            btn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> در حال پردازش...';
            
            try {
                if (!serverProjectId) {
                    throw new Error('Project ID یافت نشد');
                }
                
                const url = `/cloud/projects/${serverProjectId}/servers/${serverId}/enable-ipv6`;
                console.log('📤 Sending IPv6 request to:', url);
                
                const response = await fetch(url, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': csrfToken,
                        'Accept': 'application/json'
                    }
                });
                
                const data = await response.json();
                console.log('📥 IPv6 Response:', data);
                
                if (data.success) {
                    await Swal.fire({
                        icon: 'success',
                        title: '✅ IPv6 فعال شد',
                        html: data.ipv6 ? `<p>IPv6: <code>${data.ipv6}</code></p>` : '<p>لطفاً چند دقیقه صبر کنید...</p>',
                        showConfirmButton: true,
                    });
                    location.reload();
                } else {
                    throw new Error(data.message || 'خطا در فعال‌سازی IPv6');
                }
            } catch (error) {
                console.error('❌ IPv6 Error:', error);
                Swal.fire('خطا', error.message, 'error');
            } finally {
                btn.disabled = false;
                btn.innerHTML = '<i class="ri-add-line me-1"></i> فعال‌سازی IPv6';
            }
        });
    } else {
        console.log('ℹ️ Enable IPv6 button not found (normal)');
    }
    
    console.log('✅ Network Tab Initialized Successfully');
})();
</script>